// Copyright (c) 2022 FWORKS. All Rights Reserved.

#include "DebugMenuEntry.h"
#include "HAL/IConsoleManager.h"
#include "Framework/Application/SlateApplication.h"
#include "DebugMenuLibrary.h"
#include "EngineUtils.h"
#include "DebugMenuWidget.h"
#include "DebugMenuManager.h"
#include "Runtime/Launch/Resources/Version.h"

#define LOCTEXT_NAMESPACE "DebugMenuEntry"

UDebugMenuEntry::UDebugMenuEntry()
{

}

bool UDebugMenuEntry::IsEqualEntry(UDebugMenuEntry * OtherEntry) const
{
	return this == OtherEntry;
}

const FSlateBrush* UDebugMenuEntry::GetBorderImage() const
{
	return nullptr;
}

TSharedPtr<SWidget> UDebugMenuEntry::ConstructSlateWidget(float EntryWidth, TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager, const FSlateFontInfo& EntryFont)
{
	FLinearColor ShadowColor = FLinearColor(0, 0, 0, 1.0f);
	FIntPoint ShadowOffset = FIntPoint(-2, 2);

	FString EntryName = GetDisplayStringPrefix() + GetDisplayString();
	FLinearColor TextColor = GetDisplayColor(RuntimeEntryData, MenuManager);

	TSharedRef<STextBlock> TextBlock = SNew(STextBlock)
		.Text(FText::FromString(EntryName))
		.Font(EntryFont)
		.ColorAndOpacity(TextColor)
		.ShadowColorAndOpacity(ShadowColor)
		.ShadowOffset(ShadowOffset)
		.ToolTipText(FText::FromString(EntryName));

	TSharedRef<SHorizontalBox> HBox = SNew(SHorizontalBox)
		+ SHorizontalBox::Slot()
		.HAlign(HAlign_Left)
		[
			SNew(SBox)
			.WidthOverride(EntryWidth)
			[
				TextBlock
			]
		];

	return HBox;
}

UDebugMenuEntryBlueprint::UDebugMenuEntryBlueprint()
{

}

FString UDebugMenuEntryBlueprint::GetDisplayString() const
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* DisplayStringFunc = OuterLibrary->FindFunction(FunctionName_GetDisplayString);
		if (DisplayStringFunc != nullptr)
		{
			FStructOnScope FuncParam(DisplayStringFunc);
			OuterLibrary->ProcessEvent(DisplayStringFunc, FuncParam.GetStructMemory());
			FProperty* ReturnProp = DisplayStringFunc->GetReturnProperty();
			if (ReturnProp)
			{
				if (FStrProperty* OutPropStr = CastField<FStrProperty>(ReturnProp))
				{
					FString Result = OutPropStr->GetPropertyValue_InContainer(FuncParam.GetStructMemory());
					return Result;
				}
			}
		}
	}

	return DefaultDisplayString;
}

bool UDebugMenuEntryBlueprint::OnInputConfirm(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* ExecuteFunc = OuterLibrary->FindFunction(FunctionName_OnExecute);
		if (ExecuteFunc != nullptr)
		{
			OuterLibrary->ProcessEvent(ExecuteFunc, nullptr);

			return true;
		}
	}

	return false;
}

bool UDebugMenuEntryBlueprint::OnInputLeft(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* ExecuteFunc = OuterLibrary->FindFunction(FunctionName_OnInputLeft);
		if (ExecuteFunc != nullptr)
		{
			OuterLibrary->ProcessEvent(ExecuteFunc, nullptr);
			return true;
		}
	}

	return false;
}

bool UDebugMenuEntryBlueprint::OnInputRight(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* ExecuteFunc = OuterLibrary->FindFunction(FunctionName_OnInputRight);
		if (ExecuteFunc != nullptr)
		{
			OuterLibrary->ProcessEvent(ExecuteFunc, nullptr);
			return true;
		}
	}

	return false;
}

FLinearColor UDebugMenuEntryBlueprint::GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* DisplayColorFunc = OuterLibrary->FindFunction(FunctionName_GetColor);
		if (DisplayColorFunc != nullptr)
		{
			FStructOnScope FuncParam(DisplayColorFunc);
			OuterLibrary->ProcessEvent(DisplayColorFunc, FuncParam.GetStructMemory());
			FProperty* ReturnProp = DisplayColorFunc->GetReturnProperty();
			if (ReturnProp)
			{
				if (FStructProperty* OutPropStr = CastField<FStructProperty>(ReturnProp))
				{
					FLinearColor Result = FLinearColor::White;
#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 1
					OutPropStr->GetValue_InContainer(FuncParam.GetStructMemory(), &Result);
					return Result;
#elif ENGINE_MAJOR_VERSION == 5
					// Get a pointer to the struct instance
					Result = *OutPropStr->ContainerPtrToValuePtr<FLinearColor>(FuncParam.GetStructMemory());
					return Result;
#endif
				}
			}
		}
	}

	return DefaultDisplayColor;
}

const FSlateBrush* UDebugMenuEntryTextLabel::GetBorderImage() const
{
	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager && TargetObject.Get() && TargetProperty.Get())
	{
		FDebugMenuObjPropWatch NewWatch;
		NewWatch.PropertyChain = PropertyChain;
		NewWatch.TargetObject = TargetObject;
		NewWatch.TargetProperty = TargetProperty;
		if (MenuManager->IsPropertyBeingWatched(NewWatch))
		{
			return &MenuManager->GetWatchedBackgroundBrush();
		}
	}

	return Super::GetBorderImage();
}

void UDebugMenuEntryTextLabel::OnInputToggleWatch(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	if (MenuManager && TargetObject.Get() && TargetProperty.Get())
	{
		FDebugMenuObjPropWatch NewWatch;
		NewWatch.PropertyChain = PropertyChain;
		NewWatch.TargetObject = TargetObject;
		NewWatch.TargetProperty = TargetProperty;
		NewWatch.DisplayPrefix = GetDisplayStringPrefix();
		MenuManager->ToggleObjectPropertyWatch(NewWatch);
	}
}

#undef LOCTEXT_NAMESPACE