// Copyright (c) 2022 FWORKS. All Rights Reserved.

#include "DebugMenuEntryBoolBase.h"
#include "HAL/IConsoleManager.h"
#include "Framework/Application/SlateApplication.h"
#include "DebugMenuLibrary.h"
#include "EngineUtils.h"
#include "DebugMenuWidget.h"
#include "Engine/GameViewportClient.h"
#include "Engine/GameInstance.h"
#include "Engine/World.h"
#include "Kismet/GameplayStatics.h"
#include "Kismet/KismetSystemLibrary.h"
#include "DebugMenuManager.h"
#include "TimerManager.h"

#define LOCTEXT_NAMESPACE "DebugMenuEntryBoolBase"

UDebugMenuEntryBoolBase::UDebugMenuEntryBoolBase()
{

}

FLinearColor UDebugMenuEntryBoolBase::GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const
{
	const bool CurrentValue = GetBoolValue();
	return CurrentValue ? ColorWhenTrue : ColorWhenFalse;
}

bool UDebugMenuEntryBoolBase::OnInputConfirm(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	const bool CurrentVal = GetBoolValue();
	if (CurrentVal)
	{
		return OnInputLeft(RuntimeEntryData, MenuManager);
	}
	else
	{
		return OnInputRight(RuntimeEntryData, MenuManager);
	}
}

bool UDebugMenuEntryBoolBase::OnInputCancel(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	return false;
}

bool UDebugMenuEntryBoolBase::OnInputLeft(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	const bool CurrentValue = GetBoolValue();
	if (CurrentValue)
	{
		SetBoolValue(false);
		OnValueChanged.Broadcast(false);
		return true;
	}

	return false;
}

bool UDebugMenuEntryBoolBase::OnInputRight(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	const bool CurrentValue = GetBoolValue();
	if (!CurrentValue)
	{
		SetBoolValue(true);
		OnValueChanged.Broadcast(true);
		return true;
	}

	return false;
}

UDebugMenuEntryBoolCVar::UDebugMenuEntryBoolCVar()
{

}

FString UDebugMenuEntryBoolCVar::GetDisplayString() const
{
	FString Result = DisplayString;
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*CVarString);
	if (CVar == nullptr)
	{
		Result = Result + " (not found)";
	}

	return Result;
}

void UDebugMenuEntryBoolCVar::SetBoolValue(bool NewValue)
{
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*CVarString);
	if (CVar != nullptr)
	{
		CVar->Set(NewValue, ECVF_SetByConsole);
	}
}

bool UDebugMenuEntryBoolCVar::GetBoolValue() const
{
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*CVarString);
	if (CVar != nullptr)
	{
		return CVar->GetBool();
	}

	return false;
}

bool UDebugMenuEntryBoolCVar::ShouldShowEntry() const
{
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*CVarString);
	if (CVar == nullptr)
	{
		return false;
	}

	return true;
}

FLinearColor UDebugMenuEntryBoolCVar::GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const
{
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*CVarString);
	if (CVar)
	{
		const bool CurrentValue = CVar->GetBool();
		return CurrentValue ? ColorWhenTrue : ColorWhenFalse;
	}

	return ColorWhenInvalid;
}

UDebugMenuEntryBoolEngineStat::UDebugMenuEntryBoolEngineStat()
{

}

FString UDebugMenuEntryBoolEngineStat::GetDisplayString() const
{
	return DisplayString;
}

void UDebugMenuEntryBoolEngineStat::SetBoolValue(bool NewValue)
{
	if (NewValue != GetBoolValue())
	{
		FString Command = "stat " + StatString;
		UKismetSystemLibrary::ExecuteConsoleCommand(GetWorld(), Command, nullptr);
	}
}

bool UDebugMenuEntryBoolEngineStat::GetBoolValue() const
{
	UWorld* World = GetWorld();
	if (World)
	{
		UGameInstance* GameInst = UGameplayStatics::GetGameInstance(World);
		if (GameInst)
		{
			UGameViewportClient* ViewportClient = GameInst->GetGameViewportClient();
			if (ViewportClient)
			{
				return ViewportClient->IsStatEnabled(StatString);
			}
		}
	}

	return false;
}

UDebugMenuEntryBoolShowFlag::UDebugMenuEntryBoolShowFlag()
{

}

FString UDebugMenuEntryBoolShowFlag::GetDisplayString() const
{
	return DisplayString;
}

void UDebugMenuEntryBoolShowFlag::SetBoolValue(bool NewValue)
{
	if (NewValue)
	{
		FString Command = "showflag." + ShowFlagString + " 1";
		UKismetSystemLibrary::ExecuteConsoleCommand(GetWorld(), Command, nullptr);
	}
	else
	{
		FString Command = "showflag." + ShowFlagString + " 0";
		UKismetSystemLibrary::ExecuteConsoleCommand(GetWorld(), Command, nullptr);
	}
}

bool UDebugMenuEntryBoolShowFlag::GetBoolValue() const
{
	FString ShowFlagFullString = "showflag." + ShowFlagString;
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*ShowFlagFullString);
	if (CVar)
	{
		return (bool) CVar->GetInt();
	}

	return false;
}

FString UDebugMenuEntryBoolBlueprint::GetDisplayString() const
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* DisplayStringFunc = OuterLibrary->FindFunction(FunctionName_GetDisplayString);
		if (DisplayStringFunc != nullptr)
		{
			FStructOnScope FuncParam(DisplayStringFunc);
			OuterLibrary->ProcessEvent(DisplayStringFunc, FuncParam.GetStructMemory());
			FProperty* ReturnProp = DisplayStringFunc->GetReturnProperty();
			if (ReturnProp)
			{
				if (FStrProperty* OutPropStr = CastField<FStrProperty>(ReturnProp))
				{
					FString Result = OutPropStr->GetPropertyValue_InContainer(FuncParam.GetStructMemory());
					return Result + ": ";
				}
			}
		}
	}

	return DefaultDisplayString;
}

bool UDebugMenuEntryBoolBlueprint::GetBoolValue() const
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* BoolGetterFunc = OuterLibrary->FindFunction(FunctionName_GetBoolValue);
		if (BoolGetterFunc != nullptr)
		{
			FStructOnScope FuncParam(BoolGetterFunc);
			OuterLibrary->ProcessEvent(BoolGetterFunc, FuncParam.GetStructMemory());
			FProperty* ReturnProp = BoolGetterFunc->GetReturnProperty();
			if (ReturnProp)
			{
				if (FBoolProperty* OutPropInt = CastField<FBoolProperty>(ReturnProp))
				{
					bool Result = OutPropInt->GetPropertyValue_InContainer(FuncParam.GetStructMemory());
					return Result;
				}
			}
		}
	}

	return 0.0f;
}

void UDebugMenuEntryBoolBlueprint::SetBoolValue(bool NewValue)
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* BoolSetterFunc = OuterLibrary->FindFunction(FunctionName_SetBoolValue);
		if (BoolSetterFunc != nullptr)
		{
			FStructOnScope FuncParam(BoolSetterFunc);
			FBoolProperty* InProp = CastField<FBoolProperty>(BoolSetterFunc->FindPropertyByName(TEXT("Input")));
			if (InProp)
			{
				InProp->SetPropertyValue_InContainer(FuncParam.GetStructMemory(), NewValue);
				OuterLibrary->ProcessEvent(BoolSetterFunc, FuncParam.GetStructMemory());
			}
		}
	}
}


UDebugMenuEntryBoolUObjectProp::UDebugMenuEntryBoolUObjectProp()
{

}

FString UDebugMenuEntryBoolUObjectProp::GetDisplayString() const
{
	if (TargetObject.IsValid())
	{
		if (FProperty* TargetPropertyRaw = TargetProperty.Get())
		{
			return TargetPropertyRaw->GetName();
		}
	}

	return "(Unknown)";
}

void UDebugMenuEntryBoolUObjectProp::OnInputToggleWatch(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	if (MenuManager)
	{
		FDebugMenuObjPropWatch NewWatch;
		NewWatch.PropertyChain = PropertyChain;
		NewWatch.TargetObject = TargetObject;
		NewWatch.TargetProperty = TargetProperty;
		NewWatch.DisplayPrefix = GetDisplayStringPrefix();
		MenuManager->ToggleObjectPropertyWatch(NewWatch);
	}
}

const FSlateBrush* UDebugMenuEntryBoolUObjectProp::GetBorderImage() const
{
	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager)
	{
		FDebugMenuObjPropWatch NewWatch;
		NewWatch.PropertyChain = PropertyChain;
		NewWatch.TargetObject = TargetObject;
		NewWatch.TargetProperty = TargetProperty;
		if (MenuManager->IsPropertyBeingWatched(NewWatch))
		{
			return &MenuManager->GetWatchedBackgroundBrush();
		}
	}

	return Super::GetBorderImage();
}

bool UDebugMenuEntryBoolUObjectProp::GetBoolValue() const
{
	UObject* TargetObjectRaw = TargetObject.Get();
	FProperty* TargetPropertyRaw = TargetProperty.Get();
	if (TargetObjectRaw && TargetPropertyRaw)
	{
		if (FBoolProperty* OutPropBool = CastField<FBoolProperty>(TargetPropertyRaw))
		{
			void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObjectRaw, PropertyChain);
			bool Result = OutPropBool->GetPropertyValue_InContainer(ContainerAddress);
			return Result;
		}
	}

	return false;
}

void UDebugMenuEntryBoolUObjectProp::SetBoolValue(bool NewValue)
{
	UObject* TargetObjectRaw = TargetObject.Get();
	FProperty* TargetPropertyRaw = TargetProperty.Get();
	if (TargetObjectRaw && TargetPropertyRaw)
	{
		if (FBoolProperty* OutPropBool = CastField<FBoolProperty>(TargetPropertyRaw))
		{
			void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObjectRaw, PropertyChain);
			OutPropBool->SetPropertyValue_InContainer(ContainerAddress, NewValue, 0);
		}
	}
}

bool UDebugMenuEntryBoolUObjectProp::IsEqualEntry(UDebugMenuEntry* OtherEntry) const
{
	if (Super::IsEqualEntry(OtherEntry))
	{
		return true;
	}

	if (UDebugMenuEntryBoolUObjectProp* OtherPropEntry = Cast< UDebugMenuEntryBoolUObjectProp>(OtherEntry))
	{
		if (TargetObject != OtherPropEntry->TargetObject)
		{
			return false;
		}

		if (TargetProperty != OtherPropEntry->TargetProperty)
		{
			return false;
		}

		if (PropertyChain.Num() != OtherPropEntry->PropertyChain.Num())
		{
			return false;
		}

		for (int32 i = 0; i < PropertyChain.Num(); i++)
		{
			if (PropertyChain[i].ArrayIndex != OtherPropEntry->PropertyChain[i].ArrayIndex)
			{
				return false;
			}

			if (PropertyChain[i].Property != OtherPropEntry->PropertyChain[i].Property)
			{
				return false;
			}
		}

		return true;
	}

	return false;
}

#undef LOCTEXT_NAMESPACE
