// Copyright (c) 2022 FWORKS. All Rights Reserved.

#include "DebugMenuEntryEnumUObjectProp.h"
#include "HAL/IConsoleManager.h"
#include "Framework/Application/SlateApplication.h"
#include "DebugMenuLibrary.h"
#include "EngineUtils.h"
#include "DebugMenuWidget.h"
#include "DebugMenuManager.h"
#include "Runtime/Launch/Resources/Version.h"

#define LOCTEXT_NAMESPACE "DebugMenuEntryEnum"

UDebugMenuEntryEnumUObjectProp::UDebugMenuEntryEnumUObjectProp()
{

}

bool UDebugMenuEntryEnumUObjectProp::OnInputLeft(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	if (UObject* TargetObjectRaw = TargetObject.Get())
	{
		if (FEnumProperty* PropEnum = CastField<FEnumProperty>(TargetProperty.Get()))
		{
			void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObjectRaw, PropertyChain);
			void* ValuePtr = PropEnum->ContainerPtrToValuePtr<void>(ContainerAddress);
			FNumericProperty* UnderlyingProperty = PropEnum->GetUnderlyingProperty();
			int64 EnumValue = UnderlyingProperty->GetSignedIntPropertyValue(ValuePtr);
			UEnum* Enum = PropEnum->GetEnum();
			if (Enum)
			{
				int64 NewValue = MIN_int64;
				bool bNewValueFound = false;
				// find the biggest value smaller than the current value
				for (int32 i = 0; i < Enum->NumEnums(); i++)
				{
					int64 TempValue = Enum->GetValueByIndex(i);
					if (TempValue < EnumValue && TempValue > NewValue)
					{
						bNewValueFound = true;
						NewValue = TempValue;
					}
				}

				if (bNewValueFound)
				{
					UnderlyingProperty->SetIntPropertyValue(ValuePtr, NewValue);
					return true;
				}
			}
		}
	}

	return false;
}

bool UDebugMenuEntryEnumUObjectProp::OnInputRight(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	if (UObject* TargetObjectRaw = TargetObject.Get())
	{
		if (FEnumProperty* PropEnum = CastField<FEnumProperty>(TargetProperty.Get()))
		{
			void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObjectRaw, PropertyChain);
			void* ValuePtr = PropEnum->ContainerPtrToValuePtr<void>(ContainerAddress);
			FNumericProperty* UnderlyingProperty = PropEnum->GetUnderlyingProperty();
			int64 EnumValue = UnderlyingProperty->GetSignedIntPropertyValue(ValuePtr);
			UEnum* Enum = PropEnum->GetEnum();
			if (Enum)
			{
				int64 NewValue = MAX_int64;
				bool bNewValueFound = false;
				// find the smallest value bigger than the current value
				for (int32 i = 0; i < Enum->NumEnums(); i++)
				{
					int64 TempValue = Enum->GetValueByIndex(i);
					if (TempValue > EnumValue && TempValue < NewValue)
					{
						bNewValueFound = true;
						NewValue = TempValue;
					}
				}

				if (bNewValueFound)
				{
					UnderlyingProperty->SetIntPropertyValue(ValuePtr, NewValue);
					return true;
				}
			}
		}
	}

	return false;
}

const FSlateBrush* UDebugMenuEntryEnumUObjectProp::GetBorderImage() const
{
	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager)
	{
		FDebugMenuObjPropWatch NewWatch;
		NewWatch.PropertyChain = PropertyChain;
		NewWatch.TargetObject = TargetObject;
		NewWatch.TargetProperty = TargetProperty;
		if (MenuManager->IsPropertyBeingWatched(NewWatch))
		{
			return &MenuManager->GetWatchedBackgroundBrush();
		}
	}

	return Super::GetBorderImage();
}

void UDebugMenuEntryEnumUObjectProp::OnInputToggleWatch(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	if (MenuManager)
	{
		FDebugMenuObjPropWatch NewWatch;
		NewWatch.PropertyChain = PropertyChain;
		NewWatch.TargetObject = TargetObject;
		NewWatch.TargetProperty = TargetProperty;
		NewWatch.DisplayPrefix = GetDisplayStringPrefix();
		MenuManager->ToggleObjectPropertyWatch(NewWatch);
	}
}

FString UDebugMenuEntryEnumUObjectProp::GetDisplayString() const
{
	if (UObject* TargetObjectRaw = TargetObject.Get())
	{
		if (FEnumProperty* PropEnum = CastField<FEnumProperty>(TargetProperty.Get()))
		{
			void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObjectRaw, PropertyChain);
			void* ValuePtr = PropEnum->ContainerPtrToValuePtr<void>(ContainerAddress);
			FNumericProperty* UnderlyingProperty = PropEnum->GetUnderlyingProperty();
			int64 EnumValue = UnderlyingProperty->GetSignedIntPropertyValue(ValuePtr);
			FString ValueStr = "";
			UEnum* Enum = PropEnum->GetEnum();
			if (Enum)
			{
				if (Enum->IsValidEnumValue(EnumValue))
				{
					ValueStr = Enum->GetDisplayNameTextByValue(EnumValue).ToString();
				}
				else
				{
					ValueStr = "INVALID";
				}
			}
			else
			{
				ValueStr = FText::AsNumber(EnumValue).ToString();
			}

			return PropEnum->GetName() + ": << " + ValueStr + " >>";
		}
	}

	return "(Unknown)";
}

FLinearColor UDebugMenuEntryEnumUObjectProp::GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const
{
	return FLinearColor::White;
}

bool UDebugMenuEntryEnumUObjectProp::IsEqualEntry(UDebugMenuEntry* OtherEntry) const
{
	if (Super::IsEqualEntry(OtherEntry))
	{
		return true;
	}

	if (UDebugMenuEntryEnumUObjectProp* OtherPropEntry = Cast< UDebugMenuEntryEnumUObjectProp>(OtherEntry))
	{
		if (TargetObject != OtherPropEntry->TargetObject)
		{
			return false;
		}

		if (TargetProperty != OtherPropEntry->TargetProperty)
		{
			return false;
		}

		if (PropertyChain.Num() != OtherPropEntry->PropertyChain.Num())
		{
			return false;
		}

		for (int32 i = 0; i < PropertyChain.Num(); i++)
		{
			if (PropertyChain[i].ArrayIndex != OtherPropEntry->PropertyChain[i].ArrayIndex)
			{
				return false;
			}

			if (PropertyChain[i].Property != OtherPropEntry->PropertyChain[i].Property)
			{
				return false;
			}
		}

		return true;
	}

	return false;
}


#undef LOCTEXT_NAMESPACE