// Copyright (c) 2022 FWORKS. All Rights Reserved.

#include "DebugMenuEntryFloatBase.h"
#include "HAL/IConsoleManager.h"
#include "Framework/Application/SlateApplication.h"
#include "DebugMenuLibrary.h"
#include "Widgets/Input/SNumericEntryBox.h"
#include "EngineUtils.h"
#include "DebugMenuWidget.h"
#include "DebugMenuManager.h"
#include "Runtime/Launch/Resources/Version.h"

#define LOCTEXT_NAMESPACE "DebugMenuEntryFloatBase"

UDebugMenuEntryFloatBase::UDebugMenuEntryFloatBase()
{

}

bool UDebugMenuEntryFloatBase::OnInputConfirm(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	if (RuntimeEntryData->GetCurrentFocusWidget().IsValid())
	{
		TSharedPtr<SWidget> CurrentFocus = FSlateApplication::Get().GetUserFocusedWidget(0);
		if (MenuManager && MenuManager->IsWidgetInChildHierarchy(CurrentFocus, RuntimeEntryData->GetCurrentFocusWidget().Pin()))
		{
			FSlateApplication::Get().SetAllUserFocusToGameViewport();
			return true;
		}
		else
		{
			FSlateApplication::Get().SetAllUserFocus(RuntimeEntryData->GetCurrentFocusWidget().Pin());
			return false;
		}
	}
	return false;
}

bool UDebugMenuEntryFloatBase::OnInputCancel(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	if (RuntimeEntryData->GetCurrentFocusWidget().IsValid())
	{
		TSharedPtr<SWidget> CurrentFocus = FSlateApplication::Get().GetUserFocusedWidget(0);
		if (MenuManager && MenuManager->IsWidgetInChildHierarchy(CurrentFocus, RuntimeEntryData->GetCurrentFocusWidget().Pin()))
		{
			FSlateApplication::Get().SetAllUserFocusToGameViewport();
			MenuManager->RefreshMenu();
			return true;
		}
	}

	return false;
}

bool UDebugMenuEntryFloatBase::OnInputLeft(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	float CurrentValue = GetFloatValue();
	CurrentValue -= TweakInterval;
	if (bLimitRange)
	{
		CurrentValue = FMath::Clamp(CurrentValue, RangeMin, RangeMax);
	}
	SetFloatValue(CurrentValue);

	return true;
}

bool UDebugMenuEntryFloatBase::OnInputRight(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	float CurrentValue = GetFloatValue();
	CurrentValue += TweakInterval;
	if (bLimitRange)
	{
		CurrentValue = FMath::Clamp(CurrentValue, RangeMin, RangeMax);
	}
	SetFloatValue(CurrentValue);
	
	return true;
}

void UDebugMenuEntryFloatBase::OnUndeterminedValueCommited(FText NewValue, ETextCommit::Type CommitType)
{
	FSlateApplication::Get().SetAllUserFocusToGameViewport();

	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager)
	{
		MenuManager->RefreshMenu();
	}
}

TOptional<float> UDebugMenuEntryFloatBase::GetFloatBoxValue() const
{ 
	return GetFloatValue(); 
}

void UDebugMenuEntryFloatBase::OnValueCommited(float InValue, ETextCommit::Type InCommitType)
{
	if (bLimitRange)
	{
		InValue = FMath::Clamp(InValue, RangeMin, RangeMax);
	}
	SetFloatValue(InValue);

	FSlateApplication::Get().SetAllUserFocusToGameViewport();

	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager)
	{
		MenuManager->RefreshMenu();
	}
}

TSharedPtr<SWidget> UDebugMenuEntryFloatBase::ConstructSlateWidget(float EntryWidth, TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager, const FSlateFontInfo& EntryFont)
{
	FLinearColor ShadowColor = FLinearColor(0, 0, 0, 1.0f);
	FIntPoint ShadowOffset = FIntPoint(-2, 2);
	FString EntryName = GetDisplayStringPrefix() + GetDisplayString();
	FLinearColor TextColor = GetDisplayColor(RuntimeEntryData, MenuManager);

	TSharedRef<STextBlock> text = SNew(STextBlock)
		.Text(FText::FromString(EntryName))
		.Font(EntryFont)
		.ColorAndOpacity(TextColor)
		.ShadowColorAndOpacity(ShadowColor)
		.ShadowOffset(ShadowOffset)
		.ToolTipText(FText::FromString(EntryName));

	TSharedRef<SNumericEntryBox<float>> FloatBox = SNew(SNumericEntryBox<float>)
		.AllowSpin(bLimitRange)
		.MinSliderValue(RangeMin)
		.MaxSliderValue(RangeMax)
		.OnUndeterminedValueCommitted_UObject(this, &UDebugMenuEntryFloatBase::OnUndeterminedValueCommited)
		.Value_UObject(this, &UDebugMenuEntryFloatBase::GetFloatBoxValue)
		.OnValueCommitted_UObject(this, &UDebugMenuEntryFloatBase::OnValueCommited)
		.ToolTipText(LOCTEXT("FloatBoxTooltip", "Float value"));

	RuntimeEntryData->SetFocusWidgetOnConfirm(FloatBox);

	TSharedRef<SHorizontalBox> hbox = SNew(SHorizontalBox)

		+ SHorizontalBox::Slot()
		.MaxWidth(EntryWidth)
		[
			SNew(SHorizontalBox)
			+ SHorizontalBox::Slot()
			.HAlign(HAlign_Left)
			.AutoWidth()
			[
				SNew(SBox)
				[
					text
				]
			]
	
			+ SHorizontalBox::Slot()
			.HAlign(HAlign_Right)
			[
				SNew(SBox)
				.MinDesiredWidth(0.95f * EntryWidth)
				[
					FloatBox
				]
			]

		];
	return hbox;
}


UDebugMenuEntryFloatCVar::UDebugMenuEntryFloatCVar()
{

}

FString UDebugMenuEntryFloatCVar::GetDisplayString() const
{
	FString Result = DisplayString;
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*CVarString);
	if (CVar)
	{
		Result = Result + ":  ";
	}
	else
	{
		Result = Result + "(not found)";
	}

	return Result;
}

bool UDebugMenuEntryFloatCVar::ShouldShowEntry() const
{
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*CVarString);
	if (CVar == nullptr)
	{
		return false;
	}

	return true;
}


void UDebugMenuEntryFloatCVar::SetFloatValue(float NewValue)
{
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*CVarString);
	if (CVar)
	{
		CVar->Set(NewValue, ECVF_SetByConsole);
	}
}

float UDebugMenuEntryFloatCVar::GetFloatValue() const
{
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*CVarString);
	if (CVar)
	{
		return CVar->GetFloat();
	}

	return 0.0f;
}

FLinearColor UDebugMenuEntryFloatCVar::GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const
{
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*CVarString);
	if (CVar)
	{
		if (bLimitRange)
		{
			float CurrentValue = CVar->GetFloat();
			float ValueRatio = FMath::GetMappedRangeValueClamped(FVector2D(RangeMin, RangeMax), FVector2D(0.0f, 1.0f), CurrentValue);
			return FMath::Lerp(TextColorRangeMin, TextColorRangeMax, ValueRatio);
		}
		else
		{
			return TextColorRangeMin;
		}
	}

	return ColorWhenInvalid;
}

FLinearColor UDebugMenuEntryFloatBlueprint::GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* DisplayColorFunc = OuterLibrary->FindFunction(FunctionName_GetColor);
		if (DisplayColorFunc != nullptr)
		{
			FStructOnScope FuncParam(DisplayColorFunc);
			OuterLibrary->ProcessEvent(DisplayColorFunc, FuncParam.GetStructMemory());
			FProperty* ReturnProp = DisplayColorFunc->GetReturnProperty();
			if (ReturnProp)
			{
				if (FStructProperty* OutPropStr = CastField<FStructProperty>(ReturnProp))
				{
					FLinearColor Result = FLinearColor::White;
#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 1
					OutPropStr->GetValue_InContainer(FuncParam.GetStructMemory(), &Result);
					return Result;
#elif ENGINE_MAJOR_VERSION == 5
					// Get a pointer to the struct instance
					Result = *OutPropStr->ContainerPtrToValuePtr<FLinearColor>(FuncParam.GetStructMemory());
					return Result;
#endif
				}
			}
		}
	}

	return DefaultDisplayColor;
}

FString UDebugMenuEntryFloatBlueprint::GetDisplayString() const
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* DisplayStringFunc = OuterLibrary->FindFunction(FunctionName_GetDisplayString);
		if (DisplayStringFunc != nullptr)
		{
			FStructOnScope FuncParam(DisplayStringFunc);
			OuterLibrary->ProcessEvent(DisplayStringFunc, FuncParam.GetStructMemory());
			FProperty* ReturnProp = DisplayStringFunc->GetReturnProperty();
			if (ReturnProp)
			{
				if (FStrProperty* OutPropStr = CastField<FStrProperty>(ReturnProp))
				{
					FString Result = OutPropStr->GetPropertyValue_InContainer(FuncParam.GetStructMemory());
					return Result + ": ";
				}
			}
		}
	}

	return DefaultDisplayString;
}


float UDebugMenuEntryFloatBlueprint::GetFloatValue() const
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* FloatGetterFunc = OuterLibrary->FindFunction(FunctionName_GetFloatValue);
		if (FloatGetterFunc != nullptr)
		{
			FStructOnScope FuncParam(FloatGetterFunc);
			OuterLibrary->ProcessEvent(FloatGetterFunc, FuncParam.GetStructMemory());
			FProperty* ReturnProp = FloatGetterFunc->GetReturnProperty();
			if (ReturnProp)
			{
				if (FDoubleProperty* OutPropDouble = CastField<FDoubleProperty>(ReturnProp))
				{
					double Result = OutPropDouble->GetPropertyValue_InContainer(FuncParam.GetStructMemory());
					return (float)Result;
				}
			}
		}
	}

	return 0.0f;
}

void UDebugMenuEntryFloatBlueprint::SetFloatValue(float NewValue)
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* FloatSetterFunc = OuterLibrary->FindFunction(FunctionName_SetFloatValue);
		if (FloatSetterFunc != nullptr)
		{
			FStructOnScope FuncParam(FloatSetterFunc);
			FDoubleProperty* InProp = CastField<FDoubleProperty>(FloatSetterFunc->FindPropertyByName(TEXT("Input")));
			if (InProp)
			{
				InProp->SetPropertyValue_InContainer(FuncParam.GetStructMemory(), (double) NewValue);
				OuterLibrary->ProcessEvent(FloatSetterFunc, FuncParam.GetStructMemory());
			}
		}
	}
}

UDebugMenuEntryFloatUObjectProp::UDebugMenuEntryFloatUObjectProp()
{

}

FString UDebugMenuEntryFloatUObjectProp::GetDisplayString() const
{
	if (TargetObject.IsValid())
	{
		if (FProperty* TargetPropertyRaw = TargetProperty.Get())
		{
			return TargetPropertyRaw->GetName() + " ";
		}
	}

	return "(Unknown)";
}

FLinearColor UDebugMenuEntryFloatUObjectProp::GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const
{
	return FLinearColor::White;
}

const FSlateBrush* UDebugMenuEntryFloatUObjectProp::GetBorderImage() const
{
	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager)
	{
		FDebugMenuObjPropWatch NewWatch;
		NewWatch.PropertyChain = PropertyChain;
		NewWatch.TargetObject = TargetObject;
		NewWatch.TargetProperty = TargetProperty;
		if (MenuManager->IsPropertyBeingWatched(NewWatch))
		{
			return &MenuManager->GetWatchedBackgroundBrush();
		}
	}

	return Super::GetBorderImage();
}

void UDebugMenuEntryFloatUObjectProp::OnInputToggleWatch(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	if (MenuManager)
	{
		FDebugMenuObjPropWatch NewWatch;
		NewWatch.PropertyChain = PropertyChain;
		NewWatch.TargetObject = TargetObject;
		NewWatch.TargetProperty = TargetProperty;
		NewWatch.DisplayPrefix = GetDisplayStringPrefix();
		MenuManager->ToggleObjectPropertyWatch(NewWatch);
	}
}

float UDebugMenuEntryFloatUObjectProp::GetFloatValue() const
{
	UObject* TargetObjectRaw = TargetObject.Get();
	FProperty* TargetPropertyRaw = TargetProperty.Get();
	if (TargetObjectRaw && TargetPropertyRaw)
	{
		if (FFloatProperty* OutPropFloat = CastField<FFloatProperty>(TargetPropertyRaw))
		{
			void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObjectRaw, PropertyChain);
			float Result = OutPropFloat->GetPropertyValue_InContainer(ContainerAddress);
			return Result;
		}
		else if (FDoubleProperty* OutPropDouble = CastField<FDoubleProperty>(TargetPropertyRaw))
		{
			void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObjectRaw, PropertyChain);
			double Result = OutPropDouble->GetPropertyValue_InContainer(ContainerAddress);
			return (float) Result;
		}

	}
	
	return 0.0f;
}

void UDebugMenuEntryFloatUObjectProp::SetFloatValue(float NewValue)
{
	UObject* TargetObjectRaw = TargetObject.Get();
	FProperty* TargetPropertyRaw = TargetProperty.Get();
	if (TargetObjectRaw && TargetPropertyRaw)
	{
		if (FFloatProperty* OutPropFloat = CastField<FFloatProperty>(TargetPropertyRaw))
		{
			void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObjectRaw, PropertyChain);
			OutPropFloat->SetPropertyValue_InContainer(ContainerAddress, NewValue, 0);
		}
		else if (FDoubleProperty* OutPropDouble = CastField<FDoubleProperty>(TargetPropertyRaw))
		{
			void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObjectRaw, PropertyChain);
			OutPropDouble->SetPropertyValue_InContainer(ContainerAddress, NewValue, 0);
		}
	}
}

bool UDebugMenuEntryFloatUObjectProp::IsEqualEntry(UDebugMenuEntry* OtherEntry) const
{
	if (Super::IsEqualEntry(OtherEntry))
	{
		return true;
	}

	if (UDebugMenuEntryFloatUObjectProp* OtherPropEntry = Cast< UDebugMenuEntryFloatUObjectProp>(OtherEntry))
	{
		if (TargetObject != OtherPropEntry->TargetObject)
		{
			return false;
		}

		if (TargetProperty != OtherPropEntry->TargetProperty)
		{
			return false;
		}

		if (PropertyChain.Num() != OtherPropEntry->PropertyChain.Num())
		{
			return false;
		}

		for (int32 i = 0; i < PropertyChain.Num(); i++)
		{
			if (PropertyChain[i].ArrayIndex != OtherPropEntry->PropertyChain[i].ArrayIndex)
			{
				return false;
			}

			if (PropertyChain[i].Property != OtherPropEntry->PropertyChain[i].Property)
			{
				return false;
			}
		}

		return true;
	}

	return false;
}

#undef LOCTEXT_NAMESPACE
