// Copyright (c) 2022 FWORKS. All Rights Reserved.

#include "DebugMenuEntryIntBase.h"
#include "HAL/IConsoleManager.h"
#include "Framework/Application/SlateApplication.h"
#include "DebugMenuLibrary.h"
#include "Widgets/Input/SNumericEntryBox.h"
#include "EngineUtils.h"
#include "DebugMenuWidget.h"
#include "DebugMenuManager.h"
#include "Runtime/Launch/Resources/Version.h"

#define LOCTEXT_NAMESPACE "DebugMenuEntryIntBase"

UDebugMenuEntryIntBase::UDebugMenuEntryIntBase()
{

}

bool UDebugMenuEntryIntBase::OnInputConfirm(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	if (bUseBoolRepresentation)
	{
		// input will just toggle it on/off
		int32 CurrentVal = GetIntValue();
		if (CurrentVal == 0)
		{
			return OnInputRight(RuntimeEntryData, MenuManager);
		}
		else
		{
			return OnInputLeft(RuntimeEntryData, MenuManager);
		}
	}

	if (RuntimeEntryData->GetCurrentFocusWidget().IsValid())
	{
		TSharedPtr<SWidget> CurrentFocus = FSlateApplication::Get().GetUserFocusedWidget(0);
		if (MenuManager && MenuManager->IsWidgetInChildHierarchy(CurrentFocus, RuntimeEntryData->GetCurrentFocusWidget().Pin()))
		{
			FSlateApplication::Get().SetAllUserFocusToGameViewport();
			return true;
		}
		else
		{
			FSlateApplication::Get().SetAllUserFocus(RuntimeEntryData->GetCurrentFocusWidget().Pin());
			return false;
		}
	}

	return true;
}

bool UDebugMenuEntryIntBase::OnInputCancel(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	if (bUseBoolRepresentation)
	{
		return false;
	}

	if (RuntimeEntryData->GetCurrentFocusWidget().IsValid())
	{
		TSharedPtr<SWidget> CurrentFocus = FSlateApplication::Get().GetUserFocusedWidget(0);
		if (MenuManager && MenuManager->IsWidgetInChildHierarchy(CurrentFocus, RuntimeEntryData->GetCurrentFocusWidget().Pin()))
		{
			FSlateApplication::Get().SetAllUserFocusToGameViewport();
			MenuManager->RefreshMenu();
			return true;
		}
	}

	return false;
}

bool UDebugMenuEntryIntBase::OnInputLeft(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	int32 CurrentValue = GetIntValue();
	CurrentValue -= TweakInterval;
	if (bLimitRange)
	{
		CurrentValue = FMath::Clamp(CurrentValue, bUseBoolRepresentation ? 0 : RangeMin, bUseBoolRepresentation ? 1 : RangeMax);
	}
	SetIntValue(CurrentValue);

	OnValueChanged.Broadcast(CurrentValue);

	return true;
}

bool UDebugMenuEntryIntBase::OnInputRight(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	int32 CurrentValue = GetIntValue();
	CurrentValue += TweakInterval;
	if (bLimitRange)
	{
		CurrentValue = FMath::Clamp(CurrentValue, bUseBoolRepresentation ? 0 : RangeMin, bUseBoolRepresentation ? 1: RangeMax);
	}
	SetIntValue(CurrentValue);

	OnValueChanged.Broadcast(CurrentValue);
	
	return true;
}

void UDebugMenuEntryIntBase::OnUndeterminedValueCommited(FText NewValue, ETextCommit::Type CommitType)
{
	FSlateApplication::Get().SetAllUserFocusToGameViewport();

	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager)
	{
		MenuManager->RefreshMenu();
	}
}

TOptional<int32> UDebugMenuEntryIntBase::GetIntBoxValue() const
{ 
	return GetIntValue(); 
}

void UDebugMenuEntryIntBase::OnValueCommited(int32 InValue, ETextCommit::Type InCommitType)
{
	if (bLimitRange)
	{
		InValue = FMath::Clamp(InValue, bUseBoolRepresentation ? 0 : RangeMin, bUseBoolRepresentation ? 1 : RangeMax);
	}
	SetIntValue(InValue);

	OnValueChanged.Broadcast(InValue);

	FSlateApplication::Get().SetAllUserFocusToGameViewport();

	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager)
	{
		MenuManager->RefreshMenu();
	}
}

TSharedPtr<SWidget> UDebugMenuEntryIntBase::ConstructSlateWidget(float EntryWidth, TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager, const FSlateFontInfo& EntryFont)
{
	FLinearColor ShadowColor = FLinearColor(0, 0, 0, 1.0f);
	FIntPoint ShadowOffset = FIntPoint(-2, 2);
	FString EntryName = GetDisplayStringPrefix() + GetDisplayString();
	FLinearColor TextColor = GetDisplayColor(RuntimeEntryData, MenuManager);

	TSharedRef<STextBlock> text = SNew(STextBlock)
		.Text(FText::FromString(EntryName))
		.Font(EntryFont)
		.ColorAndOpacity(TextColor)
		.ShadowColorAndOpacity(ShadowColor)
		.ShadowOffset(ShadowOffset)
		.ToolTipText(FText::FromString(EntryName));

	TSharedRef<SNumericEntryBox<int32>> IntBox = SNew(SNumericEntryBox<int32>)
		.AllowSpin(bLimitRange)
		.MinSliderValue(bUseBoolRepresentation ? 0 : RangeMin)
		.MaxSliderValue(bUseBoolRepresentation ? 1 : RangeMax)
		.OnUndeterminedValueCommitted_UObject(this, &UDebugMenuEntryIntBase::OnUndeterminedValueCommited)
		.Value_UObject(this, &UDebugMenuEntryIntBase::GetIntBoxValue)
		.OnValueCommitted_UObject(this, &UDebugMenuEntryIntBase::OnValueCommited)
		.ToolTipText(LOCTEXT("IntBoxTooltip", "Int32 value"));

	RuntimeEntryData->SetFocusWidgetOnConfirm(IntBox);

	TSharedRef<SHorizontalBox> hbox = SNew(SHorizontalBox)

		+ SHorizontalBox::Slot()
		.MaxWidth(EntryWidth)
		[
			SNew(SHorizontalBox)
			+ SHorizontalBox::Slot()
			.HAlign(HAlign_Left)
			.AutoWidth()
			[
				SNew(SBox)
				[
					text
				]
			]
	
			+ SHorizontalBox::Slot()
			.HAlign(HAlign_Right)
			[
				SNew(SBox)
				.MinDesiredWidth(0.95f * EntryWidth)
				[
					IntBox
				]
			]

		];
	return hbox;
}


UDebugMenuEntryIntCVar::UDebugMenuEntryIntCVar()
{

}

FString UDebugMenuEntryIntCVar::GetDisplayString() const
{
	FString Result = DisplayString;
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*CVarString);
	if (CVar)
	{
		Result = Result + ":  ";
	}
	else
	{
		Result = Result + "(not found)";
	}

	return Result;
}

void UDebugMenuEntryIntCVar::SetIntValue(int32 NewValue)
{
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*CVarString);
	if (CVar)
	{
		CVar->Set(NewValue, ECVF_SetByConsole);
	}
}

int32 UDebugMenuEntryIntCVar::GetIntValue() const
{
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*CVarString);
	if (CVar)
	{
		return CVar->GetInt();
	}

	return 0.0f;
}

bool UDebugMenuEntryIntCVar::ShouldShowEntry() const
{
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*CVarString);
	if (CVar == nullptr)
	{
		return false;
	}

	return true;
}

FLinearColor UDebugMenuEntryIntCVar::GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const
{
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*CVarString);
	if (CVar)
	{
		if (bLimitRange)
		{
			int32 CurrentValue = CVar->GetInt();
			float ValueRatio = FMath::GetMappedRangeValueClamped(FVector2D(bUseBoolRepresentation ? 0 : RangeMin, bUseBoolRepresentation ? 1 : RangeMax), FVector2D(0.0f, 1.0f), (float) CurrentValue);
			return FMath::Lerp(TextColorRangeMin, TextColorRangeMax, ValueRatio);
		}
		else
		{
			return TextColorRangeMin;
		}
	}

	return ColorWhenInvalid;
}

FLinearColor UDebugMenuEntryIntBlueprint::GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* DisplayColorFunc = OuterLibrary->FindFunction(FunctionName_GetColor);
		if (DisplayColorFunc != nullptr)
		{
			FStructOnScope FuncParam(DisplayColorFunc);
			OuterLibrary->ProcessEvent(DisplayColorFunc, FuncParam.GetStructMemory());
			FProperty* ReturnProp = DisplayColorFunc->GetReturnProperty();
			if (ReturnProp)
			{
				if (FStructProperty* OutPropStr = CastField<FStructProperty>(ReturnProp))
				{
					FLinearColor Result = FLinearColor::White;
#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 1
					OutPropStr->GetValue_InContainer(FuncParam.GetStructMemory(), &Result);
					return Result;
#elif ENGINE_MAJOR_VERSION == 5
					// Get a pointer to the struct instance
					Result = *OutPropStr->ContainerPtrToValuePtr<FLinearColor>(FuncParam.GetStructMemory());
					return Result;
#endif
				}
			}
		}
	}

	return DefaultDisplayColor;
}

FString UDebugMenuEntryIntBlueprint::GetDisplayString() const
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* DisplayStringFunc = OuterLibrary->FindFunction(FunctionName_GetDisplayString);
		if (DisplayStringFunc != nullptr)
		{
			FStructOnScope FuncParam(DisplayStringFunc);
			OuterLibrary->ProcessEvent(DisplayStringFunc, FuncParam.GetStructMemory());
			FProperty* ReturnProp = DisplayStringFunc->GetReturnProperty();
			if (ReturnProp)
			{
				if (FStrProperty* OutPropStr = CastField<FStrProperty>(ReturnProp))
				{
					FString Result = OutPropStr->GetPropertyValue_InContainer(FuncParam.GetStructMemory());
					return Result + ": ";
				}
			}
		}
	}

	return DefaultDisplayString;
}


int32 UDebugMenuEntryIntBlueprint::GetIntValue() const
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* IntGetterFunc = OuterLibrary->FindFunction(FunctionName_GetIntValue);
		if (IntGetterFunc != nullptr)
		{
			FStructOnScope FuncParam(IntGetterFunc);
			OuterLibrary->ProcessEvent(IntGetterFunc, FuncParam.GetStructMemory());
			FProperty* ReturnProp = IntGetterFunc->GetReturnProperty();
			if (ReturnProp)
			{
				if (FIntProperty* OutPropInt = CastField<FIntProperty>(ReturnProp))
				{
					int32 Result = OutPropInt->GetPropertyValue_InContainer(FuncParam.GetStructMemory());
					return Result;
				}
			}
		}
	}

	return 0.0f;
}

void UDebugMenuEntryIntBlueprint::SetIntValue(int32 NewValue)
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* IntSetterFunc = OuterLibrary->FindFunction(FunctionName_SetIntValue);
		if (IntSetterFunc != nullptr)
		{
			FStructOnScope FuncParam(IntSetterFunc);
			FIntProperty* InProp = CastField<FIntProperty>(IntSetterFunc->FindPropertyByName(TEXT("Input")));
			if (InProp)
			{
				InProp->SetPropertyValue_InContainer(FuncParam.GetStructMemory(), NewValue);
				OuterLibrary->ProcessEvent(IntSetterFunc, FuncParam.GetStructMemory());
			}
		}
	}
}

UDebugMenuEntryIntUObjectProp::UDebugMenuEntryIntUObjectProp()
{

}

FString UDebugMenuEntryIntUObjectProp::GetDisplayString() const
{
	if (TargetObject.IsValid())
	{
		if (FProperty* TargetPropertyRaw = TargetProperty.Get())
		{
			return TargetPropertyRaw->GetName() + " ";
		}
	}

	return "(Unknown)";
}

FLinearColor UDebugMenuEntryIntUObjectProp::GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const
{
	return FLinearColor::White;
}

int32 UDebugMenuEntryIntUObjectProp::GetIntValue() const
{
	UObject* TargetObjectRaw = TargetObject.Get();
	FProperty* TargetPropertyRaw = TargetProperty.Get();
	if (TargetObjectRaw && TargetPropertyRaw)
	{
		if (FIntProperty* OutPropInt = CastField<FIntProperty>(TargetPropertyRaw))
		{
			void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObjectRaw, PropertyChain);
			int32 Result = OutPropInt->GetPropertyValue_InContainer(ContainerAddress);
			return Result;
		}
	}

	return 0.0f;
}

void UDebugMenuEntryIntUObjectProp::SetIntValue(int32 NewValue)
{
	UObject* TargetObjectRaw = TargetObject.Get();
	FProperty* TargetPropertyRaw = TargetProperty.Get();
	if (TargetObjectRaw && TargetPropertyRaw)
	{
		if (FIntProperty* OutPropInt = CastField<FIntProperty>(TargetPropertyRaw))
		{
			void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObjectRaw, PropertyChain);
			OutPropInt->SetPropertyValue_InContainer(ContainerAddress, NewValue, 0);
		}
	}
}

const FSlateBrush* UDebugMenuEntryIntUObjectProp::GetBorderImage() const
{
	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager)
	{
		FDebugMenuObjPropWatch NewWatch;
		NewWatch.PropertyChain = PropertyChain;
		NewWatch.TargetObject = TargetObject;
		NewWatch.TargetProperty = TargetProperty;
		if (MenuManager->IsPropertyBeingWatched(NewWatch))
		{
			return &MenuManager->GetWatchedBackgroundBrush();
		}
	}

	return Super::GetBorderImage();
}

bool UDebugMenuEntryIntUObjectProp::IsEqualEntry(UDebugMenuEntry* OtherEntry) const
{
	if (Super::IsEqualEntry(OtherEntry))
	{
		return true;
	}

	if (UDebugMenuEntryIntUObjectProp* OtherPropEntry = Cast< UDebugMenuEntryIntUObjectProp>(OtherEntry))
	{
		if (TargetObject != OtherPropEntry->TargetObject)
		{
			return false;
		}

		if (TargetProperty != OtherPropEntry->TargetProperty)
		{
			return false;
		}

		if (PropertyChain.Num() != OtherPropEntry->PropertyChain.Num())
		{
			return false;
		}

		for (int32 i = 0; i < PropertyChain.Num(); i++)
		{
			if (PropertyChain[i].ArrayIndex != OtherPropEntry->PropertyChain[i].ArrayIndex)
			{
				return false;
			}

			if (PropertyChain[i].Property != OtherPropEntry->PropertyChain[i].Property)
			{
				return false;
			}
		}

		return true;
	}

	return false;
}

void UDebugMenuEntryIntUObjectProp::OnInputToggleWatch(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	if (MenuManager)
	{
		FDebugMenuObjPropWatch NewWatch;
		NewWatch.PropertyChain = PropertyChain;
		NewWatch.TargetObject = TargetObject;
		NewWatch.TargetProperty = TargetProperty;
		NewWatch.DisplayPrefix = GetDisplayStringPrefix();
		MenuManager->ToggleObjectPropertyWatch(NewWatch);
	}
}

#undef LOCTEXT_NAMESPACE