// Copyright (c) 2022 FWORKS. All Rights Reserved.

#include "DebugMenuEntryObjectRef.h"
#include "HAL/IConsoleManager.h"
#include "Framework/Application/SlateApplication.h"
#include "DebugMenuLibrary.h"
#include "EngineUtils.h"
#include "DebugMenuWidget.h"
#include "Engine/GameViewportClient.h"
#include "Engine/GameInstance.h"
#include "Engine/World.h"
#include "Kismet/GameplayStatics.h"
#include "Kismet/KismetSystemLibrary.h"
#include "DebugMenuManager.h"
#include "DebugMenuLibrary.h"
#include "DebugMenuLibraryObjectEditor.h"
#include "DrawDebugHelpers.h"

#define LOCTEXT_NAMESPACE "DebugMenuEntryObjectRef"

UDebugMenuEntryObjectRef::UDebugMenuEntryObjectRef()
{

}

void UDebugMenuEntryObjectRef::SetReferencedObject(UObject* NewObj)
{
	ReferencedObject = TWeakObjectPtr<UObject>(NewObj);
}

FString UDebugMenuEntryObjectRef::GetDisplayString() const
{
	UObject* RefObj = ReferencedObject.Get();
	if (RefObj)
	{
		FString Result = DisplayPrefix;

		AActor* RefActor = Cast<AActor>(RefObj);
		if (RefActor && RefActor->GetWorld())
		{
#if WITH_EDITOR
			if (RefActor->GetWorld()->GetNetMode() == ENetMode::NM_DedicatedServer)
			{
				Result += "s|";
			}
			else if (RefActor->GetWorld()->GetNetMode() == ENetMode::NM_Client)
			{
				Result += "c" + FString::FromInt(RefActor->GetWorld()->GetOutermost()->GetPIEInstanceID()) + "|";
			}
#endif
		}

		Result += RefObj->GetName();

		if (RefObj->GetClass())
		{
			Result += " (";
			Result += RefObj->GetClass()->GetName();
			Result += ")";
		}
		return Result;
	}

	return "{ Unavailable }";
}

FLinearColor UDebugMenuEntryObjectRef::GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const
{
	UObject* RefObj = ReferencedObject.Get();
	if (RefObj)
	{
		return FLinearColor::White;
	}
	
	return FLinearColor::Gray;
}

bool UDebugMenuEntryObjectRef::OnInputConfirm(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	// spawn an objecteditor library, and immediately enter it
	ADebugMenuLibrary* OwnerLib = Cast<ADebugMenuLibrary>(GetOuter());
	UObject* RefObj = ReferencedObject.Get();
	if (RefObj && OwnerLib)
	{
		ADebugMenuLibraryObjectEditor* SubLibrary = GetWorld()->SpawnActorDeferred<ADebugMenuLibraryObjectEditor>(ADebugMenuLibraryObjectEditor::StaticClass(), FTransform::Identity);
		if (SubLibrary)
		{
			SubLibrary->SetDefaultTargetObject(RefObj);

			TArray<FName> OldHierarchy = OwnerLib->GetCategoryHierarchy();
			TArray<FName> NewHierarchy = OldHierarchy;
			NewHierarchy.Add(RefObj->GetFName());
			SubLibrary->SetCategoryHierarchy(NewHierarchy);

			// make sure the library self destructs on leaving
			SubLibrary->SetDestroyLibraryOnExit(true);

			SubLibrary->FinishSpawning(FTransform::Identity);

			int32 NewIndex = MenuManager->FindHierarchyIndex(NewHierarchy);
			if (NewIndex >= 0)
			{
				MenuManager->PushSelectionCrumb();

				MenuManager->SetCurrentHierarchyDisplayIndex(NewIndex);
			}

			return true;
		}
	}

	return false;
}

void UDebugMenuEntryObjectRef::OnEntryHighlighted()
{
#if !UE_BUILD_SHIPPING && !UE_BUILD_TEST
	UObject* RefObj = ReferencedObject.Get();
	UActorComponent* RefComp = Cast<UActorComponent>(RefObj);
	AActor* RefActor = RefComp ? RefComp->GetOwner() : Cast<AActor>(RefObj);
	if (RefActor && RefActor->GetWorld() == GetWorld())
	{
		FVector Origin, Extent;
		RefActor->GetActorBounds(true, Origin, Extent);
		DrawDebugBox(GetWorld(), Origin, Extent, FQuat::Identity, FColor::Green, false, 0.5f, 0, 3.f);
	}
#endif
}

const FSlateBrush* UDebugMenuEntryObjectRef::GetBorderImage() const
{
	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager)
	{
		FDebugMenuObjPropWatch NewWatch;
		NewWatch.PropertyChain = PropertyChain;
		NewWatch.TargetObject = TargetObject;
		NewWatch.TargetProperty = TargetProperty;
		if (MenuManager->IsPropertyBeingWatched(NewWatch))
		{
			return &MenuManager->GetWatchedBackgroundBrush();
		}
	}

	return Super::GetBorderImage();
}

void UDebugMenuEntryObjectRef::OnInputToggleWatch(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	if (MenuManager && TargetObject.Get() && TargetProperty.Get())
	{
		FDebugMenuObjPropWatch NewWatch;
		NewWatch.PropertyChain = PropertyChain;
		NewWatch.TargetObject = TargetObject;
		NewWatch.TargetProperty = TargetProperty;
		NewWatch.DisplayPrefix = GetDisplayStringPrefix();
		MenuManager->ToggleObjectPropertyWatch(NewWatch);
	}
}

#undef LOCTEXT_NAMESPACE