// Copyright (c) 2022 FWORKS. All Rights Reserved.

#include "DebugMenuEntryStringBase.h"
#include "HAL/IConsoleManager.h"
#include "Framework/Application/SlateApplication.h"
#include "DebugMenuLibrary.h"
#include "Widgets/Input/SEditableTextBox.h"
#include "EngineUtils.h"
#include "DebugMenuWidget.h"
#include "DebugMenuManager.h"
#include "Runtime/Launch/Resources/Version.h"
#include "TimerManager.h"

#define LOCTEXT_NAMESPACE "DebugMenuEntryStringBase"

UDebugMenuEntryStringBase::UDebugMenuEntryStringBase()
{

}

bool UDebugMenuEntryStringBase::OnInputConfirm(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	if (RuntimeEntryData->GetCurrentFocusWidget().IsValid())
	{
		TSharedPtr<SWidget> CurrentFocus = FSlateApplication::Get().GetUserFocusedWidget(0);
		if (CurrentFocus.IsValid() && MenuManager && MenuManager->IsWidgetInChildHierarchy(CurrentFocus, RuntimeEntryData->GetCurrentFocusWidget().Pin()))
		{
			FSlateApplication::Get().SetAllUserFocusToGameViewport();
			return true;
		}
		else
		{
			FSlateApplication::Get().SetAllUserFocus(RuntimeEntryData->GetCurrentFocusWidget().Pin());
			return false;
		}
	}

	return true;
}

bool UDebugMenuEntryStringBase::OnInputCancel(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	if (RuntimeEntryData->GetCurrentFocusWidget().IsValid())
	{
		TSharedPtr<SWidget> CurrentFocus = FSlateApplication::Get().GetUserFocusedWidget(0);
		if (MenuManager && MenuManager->IsWidgetInChildHierarchy(CurrentFocus, RuntimeEntryData->GetCurrentFocusWidget().Pin()))
		{
			FSlateApplication::Get().SetAllUserFocusToGameViewport();
			MenuManager->RefreshMenu();
			return true;
		}
	}

	return false;
}

bool UDebugMenuEntryStringBase::OnInputLeft(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	return false;
}

bool UDebugMenuEntryStringBase::OnInputRight(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	return false;
}

FText UDebugMenuEntryStringBase::GetStringBoxValue() const
{ 
	return FText::FromString(GetStringValue()); 
}

void UDebugMenuEntryStringBase::OnValueCommited(const FText& InNewText, ETextCommit::Type InCommitType)
{
	SetStringValue(InNewText.ToString());

	OnStringCommitted.Broadcast(InNewText.ToString());

	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager)
	{
		MenuManager->RefreshMenu();

		MenuManager->GetWorld()->GetTimerManager().SetTimerForNextTick([this]()
			{
				FSlateApplication::Get().SetAllUserFocusToGameViewport();
			});
	}
}

TSharedPtr<SWidget> UDebugMenuEntryStringBase::ConstructSlateWidget(float EntryWidth, TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager, const FSlateFontInfo& EntryFont)
{
	FLinearColor ShadowColor = FLinearColor(0, 0, 0, 1.0f);
	FIntPoint ShadowOffset = FIntPoint(-2, 2);
	FString EntryName = GetDisplayStringPrefix() + GetDisplayString();
	FLinearColor TextColor = GetDisplayColor(RuntimeEntryData, MenuManager);

	TSharedRef<STextBlock> text = SNew(STextBlock)
		.Text(FText::FromString(EntryName))
		.Font(EntryFont)
		.ColorAndOpacity(TextColor)
		.ShadowColorAndOpacity(ShadowColor)
		.ShadowOffset(ShadowOffset)
		.ToolTipText(FText::FromString(EntryName));

	TSharedRef<SEditableTextBox> StringBox = SNew(SEditableTextBox)
		.Text_UObject(this, &UDebugMenuEntryStringBase::GetStringBoxValue)
		.OnTextCommitted_UObject(this, &UDebugMenuEntryStringBase::OnValueCommited)
		.Font(EntryFont);

	StringBox->SetRevertTextOnEscape(true);
	RuntimeEntryData->SetFocusWidgetOnConfirm(StringBox);

	TSharedRef<SHorizontalBox> hbox = SNew(SHorizontalBox)

		+ SHorizontalBox::Slot()
		.MaxWidth(EntryWidth)
		[
			SNew(SHorizontalBox)
			+ SHorizontalBox::Slot()
			.HAlign(HAlign_Left)
			.AutoWidth()
			[
				SNew(SBox)
				[
					text
				]
			]
	
			+ SHorizontalBox::Slot()
			.HAlign(HAlign_Right)
			[
				SNew(SBox)
				.MinDesiredWidth(0.95f * EntryWidth)
				[
					StringBox
				]
			]

		];
	return hbox;
}

UDebugMenuEntryStringCVar::UDebugMenuEntryStringCVar()
{

}

FString UDebugMenuEntryStringCVar::GetDisplayString() const
{
	FString Result = DisplayString;
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*CVarString);
	if (CVar)
	{
		Result = Result + ":  ";
	}
	else
	{
		Result = Result + "(not found)";
	}

	return Result;
}

void UDebugMenuEntryStringCVar::SetStringValue(FString NewValue)
{
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*CVarString);
	if (CVar)
	{
		CVar->Set(*NewValue, ECVF_SetByConsole);
	}
}

FString UDebugMenuEntryStringCVar::GetStringValue() const
{
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*CVarString);
	if (CVar)
	{
		return CVar->GetString();
	}

	return "";
}

FLinearColor UDebugMenuEntryStringCVar::GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const
{
	IConsoleVariable* CVar = IConsoleManager::Get().FindConsoleVariable(*CVarString);
	if (CVar)
	{
		return Super::GetDisplayColor(RuntimeEntryData, MenuManager);
	}

	return ColorWhenInvalid;
}

UDebugMenuEntryStringUObjectProp::UDebugMenuEntryStringUObjectProp()
{

}

FString UDebugMenuEntryStringUObjectProp::GetDisplayString() const
{
	if (TargetObject.IsValid())
	{
		if (FProperty* TargetPropertyRaw = TargetProperty.Get())
		{
			return TargetPropertyRaw->GetName() + " ";
		}
	}

	return "(Unknown)";
}

FLinearColor UDebugMenuEntryStringUObjectProp::GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const
{
	return FLinearColor::White;
}

FString UDebugMenuEntryStringUObjectProp::GetStringValue() const
{
	UObject* TargetObjectRaw = TargetObject.Get();
	FProperty* TargetPropertyRaw = TargetProperty.Get();
	if (TargetObjectRaw && TargetPropertyRaw)
	{
		if (FStrProperty* OutPropStr = CastField<FStrProperty>(TargetPropertyRaw))
		{
			void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObjectRaw, PropertyChain);
			const FString Result = OutPropStr->GetPropertyValue_InContainer(ContainerAddress);
			return Result;
		}
	}

	return "";
}

void UDebugMenuEntryStringUObjectProp::SetStringValue(FString NewValue)
{
	UObject* TargetObjectRaw = TargetObject.Get();
	FProperty* TargetPropertyRaw = TargetProperty.Get();
	if (TargetObjectRaw && TargetPropertyRaw)
	{
		if (FStrProperty* OutPropStr = CastField<FStrProperty>(TargetPropertyRaw))
		{
			void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObjectRaw, PropertyChain);
			OutPropStr->SetPropertyValue_InContainer(ContainerAddress, NewValue, 0);
		}
	}
}

bool UDebugMenuEntryStringUObjectProp::IsEqualEntry(UDebugMenuEntry* OtherEntry) const
{
	if (Super::IsEqualEntry(OtherEntry))
	{
		return true;
	}

	if (UDebugMenuEntryStringUObjectProp* OtherPropEntry = Cast< UDebugMenuEntryStringUObjectProp>(OtherEntry))
	{
		if (TargetObject != OtherPropEntry->TargetObject)
		{
			return false;
		}

		if (TargetProperty != OtherPropEntry->TargetProperty)
		{
			return false;
		}

		if (PropertyChain.Num() != OtherPropEntry->PropertyChain.Num())
		{
			return false;
		}

		for (int32 i = 0; i < PropertyChain.Num(); i++)
		{
			if (PropertyChain[i].ArrayIndex != OtherPropEntry->PropertyChain[i].ArrayIndex)
			{
				return false;
			}

			if (PropertyChain[i].Property != OtherPropEntry->PropertyChain[i].Property)
			{
				return false;
			}
		}

		return true;
	}

	return false;
}

const FSlateBrush* UDebugMenuEntryStringUObjectProp::GetBorderImage() const
{
	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager)
	{
		FDebugMenuObjPropWatch NewWatch;
		NewWatch.PropertyChain = PropertyChain;
		NewWatch.TargetObject = TargetObject;
		NewWatch.TargetProperty = TargetProperty;
		if (MenuManager->IsPropertyBeingWatched(NewWatch))
		{
			return &MenuManager->GetWatchedBackgroundBrush();
		}
	}

	return Super::GetBorderImage();
}

void UDebugMenuEntryStringUObjectProp::OnInputToggleWatch(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	if (MenuManager)
	{
		FDebugMenuObjPropWatch NewWatch;
		NewWatch.PropertyChain = PropertyChain;
		NewWatch.TargetObject = TargetObject;
		NewWatch.TargetProperty = TargetProperty;
		NewWatch.DisplayPrefix = GetDisplayStringPrefix();
		MenuManager->ToggleObjectPropertyWatch(NewWatch);
	}
}

#undef LOCTEXT_NAMESPACE
