// Copyright (c) 2022 FWORKS. All Rights Reserved.

#include "DebugMenuEntryVectorBase.h"
#include "HAL/IConsoleManager.h"
#include "Framework/Application/SlateApplication.h"
#include "DebugMenuLibrary.h"
#include "Widgets/Input/SVectorInputBox.h"
#include "EngineUtils.h"
#include "DebugMenuWidget.h"
#include "DebugMenuManager.h"
#include "Runtime/Launch/Resources/Version.h"

#define LOCTEXT_NAMESPACE "DebugMenuEntryVectorBase"

UDebugMenuEntryVectorBase::UDebugMenuEntryVectorBase()
{

}

void UDebugMenuEntryVectorBase::OnValueCommitedX(float InValue, ETextCommit::Type InCommitType)
{
	FVector CurrentValue = GetVectorValue();
	CurrentValue.X = InValue;
	SetVectorValue(CurrentValue);

	FSlateApplication::Get().SetAllUserFocusToGameViewport();

	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager)
	{
		MenuManager->RefreshMenu();
	}
}

void UDebugMenuEntryVectorBase::OnValueCommitedY(float InValue, ETextCommit::Type InCommitType)
{
	FVector CurrentValue = GetVectorValue();
	CurrentValue.Y = InValue;
	SetVectorValue(CurrentValue);

	FSlateApplication::Get().SetAllUserFocusToGameViewport();

	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager)
	{
		MenuManager->RefreshMenu();
	}
}

void UDebugMenuEntryVectorBase::OnValueCommitedZ(float InValue, ETextCommit::Type InCommitType)
{
	FVector CurrentValue = GetVectorValue();
	CurrentValue.Z = InValue;
	SetVectorValue(CurrentValue);

	FSlateApplication::Get().SetAllUserFocusToGameViewport();

	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager)
	{
		MenuManager->RefreshMenu();
	}
}

bool UDebugMenuEntryVectorBase::OnInputConfirm(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	if (RuntimeEntryData->GetCurrentFocusWidget().IsValid())
	{
		TSharedPtr<SWidget> CurrentFocus = FSlateApplication::Get().GetUserFocusedWidget(0);
		if (MenuManager && MenuManager->IsWidgetInChildHierarchy(CurrentFocus, RuntimeEntryData->GetCurrentFocusWidget().Pin()))
		{
			FSlateApplication::Get().SetAllUserFocusToGameViewport();
			return true;
		}
		else
		{
			FSlateApplication::Get().SetAllUserFocus(RuntimeEntryData->GetCurrentFocusWidget().Pin());
			return false;
		}
	}
	return false;
}

bool UDebugMenuEntryVectorBase::OnInputCancel(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	if (RuntimeEntryData->GetCurrentFocusWidget().IsValid())
	{
		TSharedPtr<SWidget> CurrentFocus = FSlateApplication::Get().GetUserFocusedWidget(0);
		if (MenuManager && MenuManager->IsWidgetInChildHierarchy(CurrentFocus, RuntimeEntryData->GetCurrentFocusWidget().Pin()))
		{
			FSlateApplication::Get().SetAllUserFocusToGameViewport();
			MenuManager->RefreshMenu();
			return true;
		}
	}

	return false;
}

bool UDebugMenuEntryVectorBase::OnInputLeft(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	TSharedPtr<SNumericEntryBox<float>> CurrFloatWidget = StaticCastSharedPtr<SNumericEntryBox<float>>(RuntimeEntryData->GetCurrentFocusWidget().Pin());
	if (CurrFloatWidget.IsValid())
	{
		CurrFloatWidget->SetColorAndOpacity(FLinearColor::White);
	}

	RuntimeEntryData->DecrementFocusWidget();

	TSharedPtr<SNumericEntryBox<float>> CurrFloatWidget2 = StaticCastSharedPtr<SNumericEntryBox<float>>(RuntimeEntryData->GetCurrentFocusWidget().Pin());
	if (CurrFloatWidget2.IsValid())
	{
		CurrFloatWidget2->SetColorAndOpacity(FLinearColor::Yellow);
	}
	return false;
}

bool UDebugMenuEntryVectorBase::OnInputRight(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	TSharedPtr<SNumericEntryBox<float>> CurrFloatWidget = StaticCastSharedPtr<SNumericEntryBox<float>>(RuntimeEntryData->GetCurrentFocusWidget().Pin());
	if (CurrFloatWidget.IsValid())
	{
		CurrFloatWidget->SetColorAndOpacity(FLinearColor::White);
	}

	RuntimeEntryData->IncrementFocusWidget();

	TSharedPtr<SNumericEntryBox<float>> CurrFloatWidget2 = StaticCastSharedPtr<SNumericEntryBox<float>>(RuntimeEntryData->GetCurrentFocusWidget().Pin());
	if (CurrFloatWidget2.IsValid())
	{
		CurrFloatWidget2->SetColorAndOpacity(FLinearColor::Yellow);
	}
	return false;
}

void UDebugMenuEntryVectorBase::OnUndeterminedValueCommited(FText NewValue, ETextCommit::Type CommitType)
{
	FSlateApplication::Get().SetAllUserFocusToGameViewport();

	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager)
	{
		MenuManager->RefreshMenu();
	}
}

TSharedPtr<SWidget> UDebugMenuEntryVectorBase::ConstructSlateWidget(float EntryWidth, TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager, const FSlateFontInfo& EntryFont)
{
	FLinearColor ShadowColor = FLinearColor(0, 0, 0, 1.0f);
	FIntPoint ShadowOffset = FIntPoint(-2, 2);
	FString EntryName = GetDisplayStringPrefix() + GetDisplayString();
	FLinearColor TextColor = GetDisplayColor(RuntimeEntryData, MenuManager);

	TSharedRef<STextBlock> text = SNew(STextBlock)
		.Text(FText::FromString(EntryName))
		.Font(EntryFont)
		.ColorAndOpacity(TextColor)
		.ShadowColorAndOpacity(ShadowColor)
		.ShadowOffset(ShadowOffset)
		.ToolTipText(FText::FromString(EntryName));

	TSharedRef<SNumericEntryBox<float>> FloatBoxX = SNew(SNumericEntryBox<float>)
		.AllowSpin(false)
		.OnUndeterminedValueCommitted_UObject(this, &UDebugMenuEntryVectorBase::OnUndeterminedValueCommited)
		.Value_UObject(this, &UDebugMenuEntryVectorBase::GetValueX)
		.OnValueCommitted_UObject(this, &UDebugMenuEntryVectorBase::OnValueCommitedX)
		.ToolTipText(LOCTEXT("FloatBoxTooltip", "X value"));

	TSharedRef<SNumericEntryBox<float>> FloatBoxY = SNew(SNumericEntryBox<float>)
		.AllowSpin(false)
		.OnUndeterminedValueCommitted_UObject(this, &UDebugMenuEntryVectorBase::OnUndeterminedValueCommited)
		.Value_UObject(this, &UDebugMenuEntryVectorBase::GetValueY)
		.OnValueCommitted_UObject(this, &UDebugMenuEntryVectorBase::OnValueCommitedY)
		.ToolTipText(LOCTEXT("FloatBoxTooltip", "Y value"));

	TSharedRef<SNumericEntryBox<float>> FloatBoxZ = SNew(SNumericEntryBox<float>)
		.AllowSpin(false)
		.OnUndeterminedValueCommitted_UObject(this, &UDebugMenuEntryVectorBase::OnUndeterminedValueCommited)
		.Value_UObject(this, &UDebugMenuEntryVectorBase::GetValueZ)
		.OnValueCommitted_UObject(this, &UDebugMenuEntryVectorBase::OnValueCommitedZ)
		.ToolTipText(LOCTEXT("FloatBoxTooltip", "Z value"));

	RuntimeEntryData->FocusWidgetsOnConfirm.Empty();
	RuntimeEntryData->FocusWidgetsOnConfirm.Add(FloatBoxX);
	RuntimeEntryData->FocusWidgetsOnConfirm.Add(FloatBoxY);
	RuntimeEntryData->FocusWidgetsOnConfirm.Add(FloatBoxZ);

	TSharedPtr<SNumericEntryBox<float>> CurrFloatWidget = StaticCastSharedPtr<SNumericEntryBox<float>>(RuntimeEntryData->GetCurrentFocusWidget().Pin());
	if (CurrFloatWidget.IsValid())
	{
		CurrFloatWidget->SetColorAndOpacity(FLinearColor::Yellow);
	}

	float DisplayTextWidth = EntryWidth - 200.0f;
	if (DisplayTextWidth <= 0.0f)
	{
		DisplayTextWidth = 0.65f * EntryWidth;
	}

	TSharedRef<SHorizontalBox> hbox = SNew(SHorizontalBox)

		+ SHorizontalBox::Slot()
		.MaxWidth(EntryWidth)
		[
			SNew(SHorizontalBox)
			+ SHorizontalBox::Slot()
			.HAlign(HAlign_Left)
			
			.AutoWidth()
			[
				SNew(SBox)
				.MaxDesiredWidth(DisplayTextWidth)
				[
					text
				]
			]
	
			+ SHorizontalBox::Slot()
			.HAlign(HAlign_Right)
			[
				SNew(SBox)
				.MinDesiredWidth(0.95f * EntryWidth)
				[
					FloatBoxX
				]
			]

			+ SHorizontalBox::Slot()
			.HAlign(HAlign_Right)
			[
				SNew(SBox)
				.MinDesiredWidth(0.95f * EntryWidth)
				[
					FloatBoxY
				]
			]

			+ SHorizontalBox::Slot()
			.HAlign(HAlign_Right)
			[
				SNew(SBox)
				.MinDesiredWidth(0.95f * EntryWidth)
				[
					FloatBoxZ
				]
			]

		];
	return hbox;
}

FLinearColor UDebugMenuEntryVectorBlueprint::GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* DisplayColorFunc = OuterLibrary->FindFunction(FunctionName_GetColor);
		if (DisplayColorFunc != nullptr)
		{
			FStructOnScope FuncParam(DisplayColorFunc);
			OuterLibrary->ProcessEvent(DisplayColorFunc, FuncParam.GetStructMemory());
			FProperty* ReturnProp = DisplayColorFunc->GetReturnProperty();
			if (ReturnProp)
			{
				if (FStructProperty* OutPropStr = CastField<FStructProperty>(ReturnProp))
				{
					FLinearColor Result = FLinearColor::White;
#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 1
					OutPropStr->GetValue_InContainer(FuncParam.GetStructMemory(), &Result);
					return Result;
#elif ENGINE_MAJOR_VERSION == 5
					// Get a pointer to the struct instance
					Result = *OutPropStr->ContainerPtrToValuePtr<FLinearColor>(FuncParam.GetStructMemory());
					return Result;
#endif
				}
			}
		}
	}

	return DefaultDisplayColor;
}

FString UDebugMenuEntryVectorBlueprint::GetDisplayString() const
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* DisplayStringFunc = OuterLibrary->FindFunction(FunctionName_GetDisplayString);
		if (DisplayStringFunc != nullptr)
		{
			FStructOnScope FuncParam(DisplayStringFunc);
			OuterLibrary->ProcessEvent(DisplayStringFunc, FuncParam.GetStructMemory());
			FProperty* ReturnProp = DisplayStringFunc->GetReturnProperty();
			if (ReturnProp)
			{
				if (FStrProperty* OutPropStr = CastField<FStrProperty>(ReturnProp))
				{
					FString Result = OutPropStr->GetPropertyValue_InContainer(FuncParam.GetStructMemory());
					return Result + ": ";
				}
			}
		}
	}

	return DefaultDisplayString;
}


FVector UDebugMenuEntryVectorBlueprint::GetVectorValue() const
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* VecGetterFunc = OuterLibrary->FindFunction(FunctionName_GetVectorValue);
		if (VecGetterFunc != nullptr)
		{
			FStructOnScope FuncParam(VecGetterFunc);
			OuterLibrary->ProcessEvent(VecGetterFunc, FuncParam.GetStructMemory());
			FProperty* ReturnProp = VecGetterFunc->GetReturnProperty();
			if (ReturnProp)
			{
				FVector Result = FVector::ZeroVector;
#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 1
				ReturnProp->GetValue_InContainer(FuncParam.GetStructMemory(), &Result);
				return Result;
#elif ENGINE_MAJOR_VERSION == 5
				// Get a pointer to the struct instance
				Result = *ReturnProp->ContainerPtrToValuePtr<FVector>(FuncParam.GetStructMemory());
				return Result;
#endif
			}
		}
	}

	return FVector::ZeroVector;
}

void UDebugMenuEntryVectorBlueprint::SetVectorValue(FVector NewValue)
{
	ADebugMenuLibrary* OuterLibrary = Cast<ADebugMenuLibrary>(GetOuter());
	if (OuterLibrary != nullptr)
	{
		UFunction* VecSetterFunc = OuterLibrary->FindFunction(FunctionName_SetVectorValue);
		if (VecSetterFunc != nullptr)
		{
			FStructOnScope FuncParam(VecSetterFunc);
			FStructProperty* InProp = CastField<FStructProperty>(VecSetterFunc->FindPropertyByName(TEXT("Input")));
			if (InProp)
			{
				// todo: check this in ue5.1
				void* ValuePtr = InProp->ContainerPtrToValuePtr<void>(FuncParam.GetStructMemory());
				if (InProp->Struct->GetName() == "Vector")
				{
					InProp->CopyCompleteValue(ValuePtr, &NewValue);
					OuterLibrary->ProcessEvent(VecSetterFunc, FuncParam.GetStructMemory());
				}
			}
		}
	}
}




UDebugMenuEntryVectorUObjectProp::UDebugMenuEntryVectorUObjectProp()
{

}

FString UDebugMenuEntryVectorUObjectProp::GetDisplayString() const
{
	if (TargetObject.IsValid())
	{
		if (FProperty* TargetPropertyRaw = TargetProperty.Get())
		{
			return TargetPropertyRaw->GetName() + " ";
		}
	}

	return "(Unknown)";
}

FLinearColor UDebugMenuEntryVectorUObjectProp::GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const
{
	return FLinearColor::White;
}

FVector UDebugMenuEntryVectorUObjectProp::GetVectorValue() const
{
	UObject* TargetObjectRaw = TargetObject.Get();
	FProperty* TargetPropertyRaw = TargetProperty.Get();
	if (TargetObjectRaw && TargetPropertyRaw)
	{
		FStructProperty* ReturnProp = CastField<FStructProperty>(TargetPropertyRaw);
		if (ReturnProp)
		{
			void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObjectRaw, PropertyChain);

			FVector Result = FVector::ZeroVector;
#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 1
			ReturnProp->GetValue_InContainer(ContainerAddress, &Result);
			return Result;
#elif ENGINE_MAJOR_VERSION == 5
			// Get a pointer to the struct instance
			Result = *ReturnProp->ContainerPtrToValuePtr<FVector>(ContainerAddress);
			return Result;
#endif
		}
	}

	return FVector::ZeroVector;
}

void UDebugMenuEntryVectorUObjectProp::SetVectorValue(FVector NewValue)
{
	UObject* TargetObjectRaw = TargetObject.Get();
	FProperty* TargetPropertyRaw = TargetProperty.Get();
	if (TargetObjectRaw && TargetPropertyRaw)
	{
		FStructProperty* InProp = CastField<FStructProperty>(TargetPropertyRaw);
		if (InProp)
		{
			void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObjectRaw, PropertyChain);
			void* ValuePtr = InProp->ContainerPtrToValuePtr<void>(ContainerAddress);
			if (InProp->Struct->GetName() == "Vector")
			{
				InProp->CopyCompleteValue(ValuePtr, &NewValue);
			}
		}
	}
}

bool UDebugMenuEntryVectorUObjectProp::IsEqualEntry(UDebugMenuEntry* OtherEntry) const
{
	if (Super::IsEqualEntry(OtherEntry))
	{
		return true;
	}

	if (UDebugMenuEntryVectorUObjectProp* OtherPropEntry = Cast< UDebugMenuEntryVectorUObjectProp>(OtherEntry))
	{
		if (TargetObject != OtherPropEntry->TargetObject)
		{
			return false;
		}

		if (TargetProperty != OtherPropEntry->TargetProperty)
		{
			return false;
		}

		if (PropertyChain.Num() != OtherPropEntry->PropertyChain.Num())
		{
			return false;
		}

		for (int32 i = 0; i < PropertyChain.Num(); i++)
		{
			if (PropertyChain[i].ArrayIndex != OtherPropEntry->PropertyChain[i].ArrayIndex)
			{
				return false;
			}

			if (PropertyChain[i].Property != OtherPropEntry->PropertyChain[i].Property)
			{
				return false;
			}
		}

		return true;
	}

	return false;
}

const FSlateBrush* UDebugMenuEntryVectorUObjectProp::GetBorderImage() const
{
	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager)
	{
		FDebugMenuObjPropWatch NewWatch;
		NewWatch.PropertyChain = PropertyChain;
		NewWatch.TargetObject = TargetObject;
		NewWatch.TargetProperty = TargetProperty;
		if (MenuManager->IsPropertyBeingWatched(NewWatch))
		{
			return &MenuManager->GetWatchedBackgroundBrush();
		}
	}

	return Super::GetBorderImage();
}

void UDebugMenuEntryVectorUObjectProp::OnInputToggleWatch(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager)
{
	if (MenuManager)
	{
		FDebugMenuObjPropWatch NewWatch;
		NewWatch.PropertyChain = PropertyChain;
		NewWatch.TargetObject = TargetObject;
		NewWatch.TargetProperty = TargetProperty;
		NewWatch.DisplayPrefix = GetDisplayStringPrefix();
		MenuManager->ToggleObjectPropertyWatch(NewWatch);
	}
}

#undef LOCTEXT_NAMESPACE
