// Copyright (c) 2022 FWORKS. All Rights Reserved.

#include "DebugMenuLibrary.h"
#include "DebugMenuManager.h"
#include "DebugMenuEntry.h"
#include "EngineUtils.h"
#include "Kismet/GameplayStatics.h"

ADebugMenuLibrary::ADebugMenuLibrary()
{
	PrimaryActorTick.bCanEverTick = false;
}

void ADebugMenuLibrary::BeginPlay()
{
	Super::BeginPlay();

	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager)
	{
		MenuManager->RegisterLibrary(this);
	}
}

void ADebugMenuLibrary::EndPlay(const EEndPlayReason::Type EndPlayReason)
{
	if (bEntriesDisplayed)
	{
		NotifyEntriesDisplayEnd();
	}

	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager)
	{
		MenuManager->UnregisterLibrary(this);
	}

	Super::EndPlay(EndPlayReason);
}

const TArray<UDebugMenuEntry*>& ADebugMenuLibrary::GetEntries() const
{
	return MenuEntries;
}

void ADebugMenuLibrary::OnLibraryDisplayed()
{
	NotifyEntriesDisplayBegin();
}

void ADebugMenuLibrary::NotifyEntriesDisplayBegin()
{
	if (bEntriesDisplayed)
	{
		return;
	}

	bEntriesDisplayed = true;

	for (int32 j = 0; j < GetEntries().Num(); j++)
	{
		if (GetEntries()[j] == nullptr)
		{
			continue;
		}

		GetEntries()[j]->OnEntryDisplayBegin();
	}
}

void ADebugMenuLibrary::NotifyEntriesDisplayEnd()
{
	if (!bEntriesDisplayed)
	{
		return;
	}

	bEntriesDisplayed = false;

	for (int32 j = 0; j < GetEntries().Num(); j++)
	{
		if (GetEntries()[j] == nullptr)
		{
			continue;
		}

		GetEntries()[j]->OnEntryDisplayEnd();
	}
}

void ADebugMenuLibrary::OnLibraryExited()
{
	NotifyEntriesDisplayEnd();
}

const TArray<FName>& ADebugMenuLibrary::GetCategoryHierarchy() const
{
	return CategoryHierarchy;
}

void ADebugMenuLibrary::SetCategoryHierarchy(TArray<FName> NewHierarchy)
{
	CategoryHierarchy = NewHierarchy;
}

void ADebugMenuLibrary::ExecuteDebugConsoleCommand(const FString& Command, APlayerController* SpecificPlayer,
	EDebugConsoleCommandExecMode ExecuteMode)
{
	switch (ExecuteMode) {
		case EDebugConsoleCommandExecMode::ExecuteAnywhere:
			if(GetWorld()->IsNetMode(NM_Client))
			{
				UKismetSystemLibrary::ExecuteConsoleCommand(this, Command, SpecificPlayer);
			}
			UKismetSystemLibrary::ExecuteConsoleCommand(this, TEXT("f.Server ") + Command, SpecificPlayer);
			break;

		case EDebugConsoleCommandExecMode::ExecuteServerOnly:
			UKismetSystemLibrary::ExecuteConsoleCommand(this, TEXT("f.Server ") + Command, SpecificPlayer);
			break;

		case EDebugConsoleCommandExecMode::ExecuteClientOnly:
			if(GetWorld()->IsNetMode(NM_Client))
			{
				UKismetSystemLibrary::ExecuteConsoleCommand(this, Command, SpecificPlayer);
			}
			break;
	default: ;
	}
}

FName ADebugMenuLibrary::GetCategoryDisplayFName() const
{
	if (CategoryHierarchy.Num() <= 0)
	{
		FName NoName;
		return NoName;
	}

	return CategoryHierarchy[CategoryHierarchy.Num() - 1];
}

FString ADebugMenuLibrary::GetCategoryDisplayName() const
{
	if (CategoryHierarchy.Num() <= 0)
	{
		return "";
	}

	return CategoryHierarchy[CategoryHierarchy.Num() - 1].ToString();
}