// Copyright (c) 2022 FWORKS. All Rights Reserved.

#include "DebugMenuLibraryObjectEditor.h"
#include "DebugMenuManager.h"
#include "EngineUtils.h"
#include "GameFramework/PlayerController.h"
#include "DebugMenuEntryFloatBase.h"
#include "DebugMenuEntryIntBase.h"
#include "DebugMenuEntryBoolBase.h"
#include "DebugMenuEntryStringBase.h"
#include "DebugMenuEntryVectorBase.h"
#include "DebugMenuEntryRotatorBase.h"
#include "DebugMenuEntryEnumUObjectProp.h"
#include "DebugMenuEntryObjectRef.h"
#include "Kismet/GameplayStatics.h"
#include "GameFramework/Character.h"
#include "DrawDebugHelpers.h"
#include "GameplayTagContainer.h"

ADebugMenuLibraryObjectEditor::ADebugMenuLibraryObjectEditor()
{

}

void ADebugMenuLibraryObjectEditor::BeginPlay()
{
	Super::BeginPlay();

}

void ADebugMenuLibraryObjectEditor::EndPlay(const EEndPlayReason::Type EndPlayReason)
{
	MenuEntries.Empty();

	Super::EndPlay(EndPlayReason);
}

UObject* ADebugMenuLibraryObjectEditor::GetTargetObject_Implementation() const
{
	return DefaultTargetObject.Get();
}

void ADebugMenuLibraryObjectEditor::SetDefaultTargetObject(UObject* NewDefault)
{
	DefaultTargetObject = TWeakObjectPtr<UObject>(NewDefault);
}

void ADebugMenuLibraryObjectEditor::SetDestroyLibraryOnExit(bool bDestroy)
{
	bDestroyLibraryOnExit = bDestroy;
}

void ADebugMenuLibraryObjectEditor::OnLibraryExited()
{
	Super::OnLibraryExited();

	if (bDestroyLibraryOnExit)
	{
		Destroy();
	}
}

void ADebugMenuLibraryObjectEditor::CreateMenuEntryForProperty(FProperty* Property, UObject* TargetObj, FString Prefix, TArray<FDebugMenuPropertyChain> PropertyChain, FString InNameFilter)
{
	bool bPassNameFilter = true;
	if (InNameFilter.Len() > 0 && !Property->GetName().Contains(InNameFilter, ESearchCase::IgnoreCase))
	{
		bPassNameFilter = false;
	}
	
	if (FFloatProperty* PropFloat = CastField<FFloatProperty>(Property))
	{
		if (bPassNameFilter)
		{
			FName ObjName;
			UDebugMenuEntryFloatUObjectProp* NewProp = NewObject<UDebugMenuEntryFloatUObjectProp>(this, ObjName);
			if (NewProp)
			{
				NewProp->TargetObject = TWeakObjectPtr<UObject>(TargetObj);
				NewProp->TargetProperty = PropFloat;
				NewProp->PropertyChain = PropertyChain;
				NewProp->SetDisplayStringPrefix(Prefix);
				MenuEntries.Add(NewProp);
			}
		}
	}
	else if(FDoubleProperty * PropDouble = CastField<FDoubleProperty>(Property))
	{
		if (bPassNameFilter)
		{
			FName ObjName;
			UDebugMenuEntryFloatUObjectProp* NewProp = NewObject<UDebugMenuEntryFloatUObjectProp>(this, ObjName);
			if (NewProp)
			{
				NewProp->TargetObject = TWeakObjectPtr<UObject>(TargetObj);
				NewProp->TargetProperty = PropDouble;
				NewProp->PropertyChain = PropertyChain;
				NewProp->SetDisplayStringPrefix(Prefix);
				MenuEntries.Add(NewProp);
			}
		}
	}
	else if (FBoolProperty* PropBool = CastField<FBoolProperty>(Property))
	{
		if (bPassNameFilter)
		{
			FName ObjName;
			UDebugMenuEntryBoolUObjectProp* NewProp = NewObject<UDebugMenuEntryBoolUObjectProp>(this, ObjName);
			if (NewProp)
			{
				NewProp->TargetObject = TWeakObjectPtr<UObject>(TargetObj);
				NewProp->TargetProperty = PropBool;
				NewProp->PropertyChain = PropertyChain;
				NewProp->SetDisplayStringPrefix(Prefix);
				MenuEntries.Add(NewProp);
			}
		}
	}
	else if (FNameProperty* PropName = CastField<FNameProperty>(Property))
	{
		if (bPassNameFilter)
		{
			FName LabelObjName;
			UDebugMenuEntryTextLabel* LabelEntry = NewObject<UDebugMenuEntryTextLabel>(this, LabelObjName);
			if (LabelEntry)
			{
				LabelEntry->SetDisplayStringPrefix(Prefix);

				void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObj, PropertyChain);
				const FName PropVal = PropName->GetPropertyValue_InContainer(ContainerAddress);
				const FString TextString = Property->GetName() + ": " + PropVal.ToString();
				LabelEntry->SetTextLabel(TextString);
				MenuEntries.Add(LabelEntry);
			}
		}
	}
	else if (FIntProperty* PropInt = CastField<FIntProperty>(Property))
	{
		if (bPassNameFilter)
		{
			FName ObjName;
			UDebugMenuEntryIntUObjectProp* NewProp = NewObject<UDebugMenuEntryIntUObjectProp>(this, ObjName);
			if (NewProp)
			{
				NewProp->TargetObject = TWeakObjectPtr<UObject>(TargetObj);
				NewProp->TargetProperty = PropInt;
				NewProp->PropertyChain = PropertyChain;
				NewProp->SetDisplayStringPrefix(Prefix);
				MenuEntries.Add(NewProp);
			}
		}
	}
	else if (FStrProperty* PropStr = CastField<FStrProperty>(Property))
	{
		if (bPassNameFilter)
		{
			FName ObjName;
			UDebugMenuEntryStringUObjectProp* NewProp = NewObject<UDebugMenuEntryStringUObjectProp>(this, ObjName);
			if (NewProp)
			{
				NewProp->TargetObject = TWeakObjectPtr<UObject>(TargetObj);
				NewProp->TargetProperty = PropStr;
				NewProp->PropertyChain = PropertyChain;
				NewProp->SetDisplayStringPrefix(Prefix);
				MenuEntries.Add(NewProp);
			}
		}
	}
	else if (FStructProperty* PropStruct = CastField<FStructProperty>(Property))
	{
		if (PropStruct->Struct)
		{
			if (PropStruct->Struct->GetName() == "Vector")
			{
				if (bPassNameFilter)
				{
					FName ObjName;
					UDebugMenuEntryVectorUObjectProp* NewProp = NewObject<UDebugMenuEntryVectorUObjectProp>(this, ObjName);
					if (NewProp)
					{
						NewProp->TargetObject = TWeakObjectPtr<UObject>(TargetObj);
						NewProp->TargetProperty = PropStruct;
						NewProp->PropertyChain = PropertyChain;
						NewProp->SetDisplayStringPrefix(Prefix);
						MenuEntries.Add(NewProp);
					}
				}
			}
			else if (PropStruct->Struct->GetName() == "Rotator")
			{
				if (bPassNameFilter)
				{
					FName ObjName;
					UDebugMenuEntryRotatorUObjectProp* NewProp = NewObject<UDebugMenuEntryRotatorUObjectProp>(this, ObjName);
					if (NewProp)
					{
						NewProp->TargetObject = TWeakObjectPtr<UObject>(TargetObj);
						NewProp->TargetProperty = PropStruct;
						NewProp->PropertyChain = PropertyChain;
						NewProp->SetDisplayStringPrefix(Prefix);
						MenuEntries.Add(NewProp);
					}
				}
			}
			else if (PropStruct->Struct == FGameplayTagContainer::StaticStruct())
			{
				if (bPassNameFilter)
				{
					FName LabelObjName;
					UDebugMenuEntryTextLabel* LabelEntry = NewObject<UDebugMenuEntryTextLabel>(this, LabelObjName);
					if (LabelEntry)
					{
						LabelEntry->SetDisplayStringPrefix(Prefix);

						void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObj, PropertyChain);
						const FGameplayTagContainer TagContainer = *PropStruct->ContainerPtrToValuePtr<FGameplayTagContainer>(ContainerAddress);
						TArray<FGameplayTag> GameplayTags;
						TagContainer.GetGameplayTagArray(GameplayTags);

						FString TextString = Property->GetName();
						TextString += ": ";
						for (FGameplayTag& Tag : GameplayTags)
						{
							TextString += "\n   ";
							TextString += Tag.ToString();
						}

						LabelEntry->SetTextLabel(TextString);
						MenuEntries.Add(LabelEntry);
					}
				}
			}
			else if (PropStruct->Struct == FGameplayTag::StaticStruct())
			{
				if (bPassNameFilter)
				{
					FName LabelObjName;
					UDebugMenuEntryTextLabel* LabelEntry = NewObject<UDebugMenuEntryTextLabel>(this, LabelObjName);
					if (LabelEntry)
					{
						LabelEntry->SetDisplayStringPrefix(Prefix);

						void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObj, PropertyChain);
						const FGameplayTag Tag = *PropStruct->ContainerPtrToValuePtr<FGameplayTag>(ContainerAddress);
						const FString TextString = Property->GetName() + ": \n   " + Tag.ToString();
						LabelEntry->SetTextLabel(TextString);
						MenuEntries.Add(LabelEntry);
					}
				}
			}
			else
			{
				if (bPassNameFilter)
				{
					// if the struct variable passes the name filter, all its sub-properties pass the name filter
					InNameFilter = "";
				}

				FDebugMenuPropertyChain NewChainEntry;
				NewChainEntry.Property = PropStruct;
				PropertyChain.Add(NewChainEntry);
				FString NewPrefix = Prefix + Property->GetName() + ".";
				FField* Curr = PropStruct->Struct->ChildProperties;
				while (Curr != nullptr)
				{
					FProperty* SubProp = CastField<FProperty>(Curr);
					if (SubProp)
					{
						CreateMenuEntryForProperty(SubProp, TargetObj, NewPrefix, PropertyChain, InNameFilter);
					}
					Curr = Curr->Next;
				}
			}
		}
	}
	else if (FObjectPropertyBase* PropObj = CastField<FObjectPropertyBase>(Property))
	{
		if (bPassNameFilter)
		{
			void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObj, PropertyChain);
			UObject* ObjectReferenced = PropObj->GetObjectPropertyValue_InContainer(ContainerAddress);
			if (ObjectReferenced)
			{
				// for objects, we allow going into the object, but not changing the object itself
				FName ObjName;
				UDebugMenuEntryObjectRef* NewProp = NewObject<UDebugMenuEntryObjectRef>(this, ObjName);
				if (NewProp)
				{
					NewProp->SetDisplayPrefix(Property->GetName() + ": ");
					NewProp->TargetObject = TWeakObjectPtr<UObject>(TargetObj);
					NewProp->TargetProperty = PropObj;
					NewProp->PropertyChain = PropertyChain;
					NewProp->SetReferencedObject(ObjectReferenced);
					NewProp->SetDisplayStringPrefix(Prefix);
					MenuEntries.Add(NewProp);
				}
			}
			else
			{
				FName LabelObjName;
				UDebugMenuEntryTextLabel* LabelEntry = NewObject<UDebugMenuEntryTextLabel>(this, LabelObjName);
				if (LabelEntry)
				{
					LabelEntry->SetDisplayStringPrefix(Prefix);
					LabelEntry->SetTextLabel(Property->GetName() + ": [NULL]");
					LabelEntry->TargetObject = TWeakObjectPtr<UObject>(TargetObj);
					LabelEntry->TargetProperty = PropObj;
					LabelEntry->PropertyChain = PropertyChain;
					MenuEntries.Add(LabelEntry);
				}
			}
		}
	}
	else if (FEnumProperty* PropEnum = CastField<FEnumProperty>(Property))
	{
		if (bPassNameFilter)
		{
			FName ObjName;
			UDebugMenuEntryEnumUObjectProp* NewProp = NewObject<UDebugMenuEntryEnumUObjectProp>(this, ObjName);
			if (NewProp)
			{
				NewProp->TargetObject = TWeakObjectPtr<UObject>(TargetObj);
				NewProp->TargetProperty = PropEnum;
				NewProp->PropertyChain = PropertyChain;
				NewProp->SetDisplayStringPrefix(Prefix);
				MenuEntries.Add(NewProp);
			}
		}
	}
	else if (FByteProperty* PropByte = CastField<FByteProperty>(Property))
	{
		if (bPassNameFilter)
		{
			void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObj, PropertyChain);

#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 1
			const int64 EnumValue = PropByte->GetSignedIntPropertyValue_InContainer(ContainerAddress);
#else
			const int64 EnumValue = PropByte->GetPropertyValue_InContainer(ContainerAddress);
#endif
			const UEnum* Enum = PropByte->Enum;
			if (Enum && Enum->IsValidEnumValue(EnumValue))
			{
				FName LabelObjName;
				UDebugMenuEntryTextLabel* LabelEntry = NewObject<UDebugMenuEntryTextLabel>(this, LabelObjName);
				if (LabelEntry)
				{
					FString EnumStr = Enum->GetDisplayNameTextByValue(EnumValue).ToString();
					LabelEntry->SetDisplayStringPrefix(Prefix);
					LabelEntry->SetTextLabel(Property->GetName() + ": " + EnumStr);
					LabelEntry->TargetObject = TWeakObjectPtr<UObject>(TargetObj);
					LabelEntry->TargetProperty = PropByte;
					LabelEntry->PropertyChain = PropertyChain;
					MenuEntries.Add(LabelEntry);
				}
			}
			else
			{
				FName LabelObjName;
				UDebugMenuEntryTextLabel* LabelEntry = NewObject<UDebugMenuEntryTextLabel>(this, LabelObjName);
				if (LabelEntry)
				{
					LabelEntry->SetDisplayStringPrefix(Prefix);
					LabelEntry->SetTextLabel(Property->GetName() + ": " + FString::FromInt(EnumValue));
					LabelEntry->TargetObject = TWeakObjectPtr<UObject>(TargetObj);
					LabelEntry->TargetProperty = PropByte;
					LabelEntry->PropertyChain = PropertyChain;
					MenuEntries.Add(LabelEntry);
				}
			}
		}
	}
	else if (FMapProperty* MapProperty = CastField<FMapProperty>(Property))
	{
		FName LabelObjName;
		UDebugMenuEntryTextLabel* LabelEntry = NewObject<UDebugMenuEntryTextLabel>(this, LabelObjName);
		if (LabelEntry)
		{
			void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObj, PropertyChain);
			void* MapContainerAddress = MapProperty->ContainerPtrToValuePtr<void>(ContainerAddress);
			FScriptMapHelper MapHelper(MapProperty, MapContainerAddress);

			FProperty* ValueProp = MapHelper.GetValueProperty();
			FProperty* KeyProp = MapHelper.GetKeyProperty();

			FString LabelText = Property->GetName() + ": <Map>";
			LabelEntry->SetDisplayStringPrefix(Prefix);
			LabelEntry->SetTextLabel(LabelText);
			LabelEntry->TargetObject = TWeakObjectPtr<UObject>(TargetObj);
			LabelEntry->TargetProperty = Property;
			LabelEntry->PropertyChain = PropertyChain;
			MenuEntries.Add(LabelEntry);
		}
	}

	else if (FArrayProperty* ArrayProperty = CastField<FArrayProperty>(Property))
	{
		if (bPassNameFilter)
		{
			void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(TargetObj, PropertyChain);
			ContainerAddress = ArrayProperty->ContainerPtrToValuePtr<void>(ContainerAddress);

			FScriptArrayHelper Helper(ArrayProperty, ContainerAddress);
			int32 NumElements = Helper.Num();
			for (int32 i = 0; i < NumElements; i++)
			{
				if (Helper.IsValidIndex(i))
				{
					uint8* ElementRaw = Helper.GetRawPtr(i);

					bool bShowLabel = false;
					FString LabelText = "";
					if (FDoubleProperty* InnerPropDouble = CastField<FDoubleProperty>(ArrayProperty->Inner))
					{
						double Result = InnerPropDouble->GetPropertyValue_InContainer(ElementRaw);
						LabelText = Property->GetName() + "[" + FString::FromInt(i) + "]: " + FString::SanitizeFloat(Result);
						bShowLabel = true;
					}
					else if (FBoolProperty* InnerPropBool = CastField<FBoolProperty>(ArrayProperty->Inner))
					{
						bool Result = InnerPropBool->GetPropertyValue_InContainer(ElementRaw);
						LabelText = Property->GetName() + "[" + FString::FromInt(i) + "]: " + (Result ? "True" : "False");
						bShowLabel = true;
					}
					else if (FFloatProperty* InnerPropFloat = CastField<FFloatProperty>(ArrayProperty->Inner))
					{
						float Result = InnerPropFloat->GetPropertyValue_InContainer(ElementRaw);
						LabelText = Property->GetName() + "[" + FString::FromInt(i) + "]: " + FString::SanitizeFloat((float)Result);
						bShowLabel = true;
					}
					else if (FIntProperty* InnerPropInt = CastField<FIntProperty>(ArrayProperty->Inner))
					{
						int32 Result = InnerPropInt->GetPropertyValue_InContainer(ElementRaw);
						LabelText = Property->GetName() + "[" + FString::FromInt(i) + "]: " + FString::FromInt(Result);
						bShowLabel = true;
					}
					else if (FByteProperty* InnerPropByte = CastField<FByteProperty>(ArrayProperty->Inner))
					{
#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 1
						const int64 EnumValue = InnerPropByte->GetSignedIntPropertyValue_InContainer(ElementRaw);
#else
						const int64 EnumValue = PropByte->GetPropertyValue_InContainer(ElementRaw);
#endif
						const UEnum* Enum = InnerPropByte->Enum;
						if (Enum && Enum->IsValidEnumValue(EnumValue))
						{
							FString EnumStr = Enum->GetDisplayNameTextByValue(EnumValue).ToString();
							LabelText = Property->GetName() + "[" + FString::FromInt(i) + "]: " + EnumStr;
							bShowLabel = true;
						}
						else
						{
							LabelText = Property->GetName() + "[" + FString::FromInt(i) + "]: " + FString::FromInt(EnumValue);
							bShowLabel = true;
						}
					}
					else if (FStrProperty* InnerPropStr = CastField<FStrProperty>(ArrayProperty->Inner))
					{
						const FString Result = InnerPropStr->GetPropertyValue_InContainer(ElementRaw);
						LabelText = Property->GetName() + "[" + FString::FromInt(i) + "]: " + Result;
						bShowLabel = true;
					}
					else if (FStructProperty* InnerPropStruct = CastField<FStructProperty>(ArrayProperty->Inner))
					{
						ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
						if (MenuManager)
						{
							TArray<FDebugMenuPropertyChain> TempChain = PropertyChain;
							FDebugMenuPropertyChain NewEntry;
							NewEntry.ArrayIndex = i;
							NewEntry.Property = ArrayProperty;
							TempChain.Add(NewEntry);
							const FString Result = MenuManager->GetObjectPropertyValueAsString(TargetObj, InnerPropStruct, " ", TempChain);
							LabelText = Property->GetName() + "[" + FString::FromInt(i) + "]: " + Result;
							bShowLabel = true;
						}
					}
					else if (FObjectPropertyBase* InnerPropObj = CastField<FObjectPropertyBase>(ArrayProperty->Inner))
					{
						UObject* ObjectReferenced = InnerPropObj->GetObjectPropertyValue_InContainer(ElementRaw);
						if (ObjectReferenced)
						{
							// for objects, we allow going into the object, but not changing the object itself
							FName ObjName;
							UDebugMenuEntryObjectRef* NewProp = NewObject<UDebugMenuEntryObjectRef>(this, ObjName);
							if (NewProp)
							{
								NewProp->SetDisplayPrefix(Property->GetName() + "[" + FString::FromInt(i) + "]: ");
								NewProp->SetReferencedObject(ObjectReferenced);
								NewProp->SetDisplayStringPrefix(Prefix);
								NewProp->TargetObject = TWeakObjectPtr<UObject>(TargetObj);
								NewProp->TargetProperty = Property;
								NewProp->PropertyChain = PropertyChain;
								MenuEntries.Add(NewProp);
							}
						}
						else
						{
							FName LabelObjName;
							UDebugMenuEntryTextLabel* LabelEntry = NewObject<UDebugMenuEntryTextLabel>(this, LabelObjName);
							if (LabelEntry)
							{
								LabelEntry->SetDisplayStringPrefix(Prefix);
								LabelEntry->SetTextLabel(Property->GetName() + "[" + FString::FromInt(i) + "]: [NULL]");
								LabelEntry->TargetObject = TWeakObjectPtr<UObject>(TargetObj);
								LabelEntry->TargetProperty = Property;
								LabelEntry->PropertyChain = PropertyChain;
								MenuEntries.Add(LabelEntry);
							}
						}
					}

					if (bShowLabel)
					{
						FName LabelObjName;
						UDebugMenuEntryTextLabel* LabelEntry = NewObject<UDebugMenuEntryTextLabel>(this, LabelObjName);
						if (LabelEntry)
						{
							LabelEntry->SetDisplayStringPrefix(Prefix);
							LabelEntry->SetTextLabel(LabelText);
							LabelEntry->TargetObject = TWeakObjectPtr<UObject>(TargetObj);
							LabelEntry->TargetProperty = Property;
							LabelEntry->PropertyChain = PropertyChain;
							MenuEntries.Add(LabelEntry);
						}
					}
				}
			}
		}
	}
}

void ADebugMenuLibraryObjectEditor::RefreshProperties()
{
	UObject* TargetObj = GetTargetObject();
	if (TargetObj)
	{
		if (TargetObj != LastDisplayedObject.Get())
		{
			NotifyEntriesDisplayEnd();
			MenuEntries.Empty();

			for (TFieldIterator<FProperty> PropIt(TargetObj->GetClass()); PropIt; ++PropIt)
			{
				FProperty* Property = *PropIt;
				if (Property)
				{
#if WITH_EDITORONLY_DATA
					if (LimitToCategories.Num() > 0)
					{
						if (!Property->HasMetaData(TEXT("Category")))
						{
							continue;
						}

						bool bCategoryValid = false;
						const FString& CategoryStr = Property->GetMetaData(TEXT("Category"));
						for (FString LimitCat : LimitToCategories)
						{
							if (LimitCat.Equals(CategoryStr, ESearchCase::IgnoreCase))
							{
								bCategoryValid = true;
								break;
							}
						}

						if (!bCategoryValid)
						{
							continue;
						}
					}
#endif

					TArray<FDebugMenuPropertyChain> PropertyChain;
					CreateMenuEntryForProperty(Property, TargetObj, "", PropertyChain, NameFilter);
				}
			}

			if (bSortAlphabetically)
			{
				MenuEntries.Sort([](const UDebugMenuEntry& Lhs, const UDebugMenuEntry& Rhs) {
					FString LeftString = Lhs.GetDisplayStringPrefix() + Lhs.GetDisplayString();
					FString RightString = Rhs.GetDisplayStringPrefix() + Rhs.GetDisplayString();
					return LeftString.Compare(RightString, ESearchCase::IgnoreCase) < 0;
					});
			}

			// for actors, let's list out all their components too
			AActor* TargetActor = Cast<AActor>(TargetObj);
			if (TargetActor)
			{
				for (UActorComponent* ActorComp : TargetActor->GetComponents())
				{
					if (ActorComp == nullptr)
					{
						continue;
					}

					if (NameFilter.Len() > 0)
					{
						if (!ActorComp->GetName().Contains(NameFilter, ESearchCase::IgnoreCase))
						{
							continue;
						}
					}

					FName ObjName;
					UDebugMenuEntryObjectRef* NewProp = NewObject<UDebugMenuEntryObjectRef>(this, ObjName);
					if (NewProp)
					{
						NewProp->SetDisplayPrefix("Comp: ");
						NewProp->SetReferencedObject(ActorComp);
						MenuEntries.Insert(NewProp, 0);
					}
				}
			}

			// quick commands
			if (TargetActor && TargetActor->GetWorld() && TargetActor->GetWorld()->IsGameWorld())
			{
				FName DestroyActorObjName;
				UDebugMenuEntryBlueprint* BlueprintEntry = NewObject<UDebugMenuEntryBlueprint>(this, DestroyActorObjName);
				if (BlueprintEntry)
				{
					BlueprintEntry->DefaultDisplayString = "DESTROY Actor";
					BlueprintEntry->FunctionName_OnExecute = FName(TEXT("OnDestroyActor"));
					BlueprintEntry->DefaultDisplayColor = FLinearColor::Red;
					MenuEntries.Insert(BlueprintEntry, 0);
				}

				FName DrawDebugLineObjName;
				BlueprintEntry = NewObject<UDebugMenuEntryBlueprint>(this, DrawDebugLineObjName);
				if (BlueprintEntry)
				{
					BlueprintEntry->DefaultDisplayString = "DRAWDEBUG Line From Player";
					BlueprintEntry->FunctionName_OnExecute = FName(TEXT("OnDrawDebugLineFromPlayer"));
					MenuEntries.Insert(BlueprintEntry, 0);
				}
			}

			{
				FName LabelObjName;
				UDebugMenuEntryTextLabel* LabelEntry = NewObject<UDebugMenuEntryTextLabel>(this, LabelObjName);
				if (LabelEntry)
				{
					LabelEntry->SetTextLabel("------------------");
					MenuEntries.Insert(LabelEntry, 0);
				}
			}

			{
				FString PropertyName = "NameFilter";
				FProperty* FoundProperty = FindFProperty<FStrProperty>(this->GetClass(), *PropertyName);
				if (FoundProperty)
				{
					FName ObjName;
					UDebugMenuEntryStringUObjectProp* NewProp = NewObject<UDebugMenuEntryStringUObjectProp>(this, ObjName);
					if (NewProp)
					{
						NewProp->TargetObject = TWeakObjectPtr<UObject>(this);
						NewProp->TargetProperty = FoundProperty;
						NewProp->OnStringCommitted.AddDynamic(this, &ADebugMenuLibraryObjectEditor::OnNameFilterUpdated);
						MenuEntries.Insert(NewProp, 0);
					}
				}
			}

			FName LabelObjName;
			UDebugMenuEntryTextLabel* NameLabelEntry = NewObject<UDebugMenuEntryTextLabel>(this, LabelObjName);
			if (NameLabelEntry)
			{
				NameLabelEntry->SetTextLabel("OBJECT: " + TargetObj->GetName());
				MenuEntries.Insert(NameLabelEntry, 0);
			}

			NotifyEntriesDisplayBegin();

		}
	}
	else
	{
		NotifyEntriesDisplayEnd();
		MenuEntries.Empty();
	}

	LastDisplayedObject = TWeakObjectPtr<UObject>(TargetObj);
}

void ADebugMenuLibraryObjectEditor::OnDestroyActor()
{
	AActor* TargetActor = Cast<AActor>(GetTargetObject());
	if (TargetActor)
	{
		TargetActor->Destroy();

		ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
		if (MenuManager)
		{
			MenuManager->InputCancel();
		}
	}
}

void ADebugMenuLibraryObjectEditor::OnDrawDebugLineFromPlayer()
{
	AActor* TargetActor = Cast<AActor>(GetTargetObject());
	if (TargetActor)
	{
		const FVector TargetPos = TargetActor->GetActorLocation();
		ACharacter* PlayerChar = UGameplayStatics::GetPlayerCharacter(this, 0);
		if (PlayerChar && (TargetActor != PlayerChar))
		{
			const FVector PlayerPos = PlayerChar->GetActorLocation();
			DrawDebugLine(GetWorld(), PlayerPos, TargetPos, FColor::Yellow, false, 3.0f, 0, 5.0f);
		}

		FVector Origin, Extent;
		TargetActor->GetActorBounds(true, Origin, Extent);
		DrawDebugBox(GetWorld(), Origin, Extent, FQuat::Identity, FColor::Yellow, false, 3.0f, 0, 5.f);
	}
}

void ADebugMenuLibraryObjectEditor::OnNameFilterUpdated(FString NewText)
{
	LastDisplayedObject.Reset();

	RefreshProperties();
}

void ADebugMenuLibraryObjectEditor::OnLibraryDisplayed()
{
	RefreshProperties();

	Super::OnLibraryDisplayed();
}
