// Copyright (c) 2022 FWORKS. All Rights Reserved.

#include "DebugMenuLibraryObjectSearch.h"
#include "DebugMenuLibraryObjectEditor.h"
#include "DebugMenuManager.h"
#include "EngineUtils.h"
#include "DebugMenuEntryStringBase.h"
#include "DebugMenuEntryBoolBase.h"
#include "DebugMenuEntryIntBase.h"
#include "DebugMenuEntryObjectRef.h"
#include "GameFramework/PlayerController.h"
#include "Components/InputComponent.h"
#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 1
#include "Engine/HitResult.h"
#endif
#include "Engine/EngineTypes.h"

ADebugMenuLibraryObjectSearch::ADebugMenuLibraryObjectSearch()
{

}

void ADebugMenuLibraryObjectSearch::OnSearchFilterStringUpdated(FString NewText)
{
	PerformSearch();
}

void ADebugMenuLibraryObjectSearch::OnSearchFilterBoolUpdated(bool NewValue)
{
	PerformSearch();
}

void ADebugMenuLibraryObjectSearch::OnSearchFilterIntegerUpdated(int32 NewValue)
{
	PerformSearch();
}

void ADebugMenuLibraryObjectSearch::PerformSearch(UObject* SpecificObj /*= nullptr*/)
{
	// remove all menu entries
	const int32 KEEPMENUENTRIES = 6;
	while (MenuEntries.Num() > KEEPMENUENTRIES)
	{
		MenuEntries.RemoveAt(KEEPMENUENTRIES);
	}

	if (SearchText.Len() <= 0)
	{
		return;
	}

	TArray<FString> SearchWords;
	SearchText.ParseIntoArray(SearchWords, TEXT(" "));

	FName LabelObjName;
	UDebugMenuEntryTextLabel* LabelEntry = NewObject<UDebugMenuEntryTextLabel>(this, LabelObjName);
	if (LabelEntry)
	{
		LabelEntry->SetTextLabel("------- Results -------");
		MenuEntries.Add(LabelEntry);
	}

	int32 SearchResultsFound = 0;
	bool bSearchParentClasses = true;

	if (SpecificObj)
	{
		FName ObjName;
		UDebugMenuEntryObjectRef* NewProp = NewObject<UDebugMenuEntryObjectRef>(this, ObjName);
		if (NewProp)
		{
			NewProp->SetReferencedObject(SpecificObj);
			MenuEntries.Add(NewProp);
		}

		SearchResultsFound++;
	}
	else
	{
		for (TObjectIterator<UObject> ObjectItr; ObjectItr; ++ObjectItr)
		{
			UObject* Object = *ObjectItr;
			if (Object == nullptr)
			{
				continue;
			}

			// exclude stuff we don't care about
			if (Object->IsA(UField::StaticClass())
				|| Object->IsA(UPackage::StaticClass())
				|| Object->IsA(UMetaData::StaticClass()))
			{
				continue;
			}

			if (Object->GetClass() == UBlueprint::StaticClass())
			{
				continue;
			}

			if (!IsValid(Object))
			{
				continue;
			}

			if (Object->HasAnyFlags(RF_ClassDefaultObject) || Object->HasAnyFlags(RF_ArchetypeObject))
			{
				continue;
			}

			if (!bSearchClassName)
			{
				const FString ObjectName = Object->GetName();
				if (!StringMatchesSearch(ObjectName, SearchWords, bExactMatch))
				{
					continue;
				}
			}
			else if (Object->GetClass())
			{
				if (bSearchParentClasses)
				{
					bool bFound = false;
					UClass* CurrClass = Object->GetClass();
					while (CurrClass != nullptr)
					{
						const FString ClassName = CurrClass->GetName();

						if (StringMatchesSearch(ClassName, SearchWords, bExactMatch))
						{
							bFound = true;
							break;
						}

						CurrClass = CurrClass->GetSuperClass();
					}

					if (!bFound)
					{
						continue;
					}
				}
				else
				{
					const FString ClassName = Object->GetClass()->GetName();
					if (!StringMatchesSearch(ClassName, SearchWords, bExactMatch))
					{
						continue;
					}
				}
			}

			if (SearchResultsFound < MaxSearchResults)
			{
				FName ObjName;
				UDebugMenuEntryObjectRef* NewProp = NewObject<UDebugMenuEntryObjectRef>(this, ObjName);
				if (NewProp)
				{
					NewProp->SetReferencedObject(Object);
					MenuEntries.Add(NewProp);
				}
			}

			SearchResultsFound++;
		}
	}

	if (LabelEntry)
	{
		FString LabelText = "------- Results: ";
		LabelText += FString::FromInt(SearchResultsFound);
		LabelText += " -------";
		LabelEntry->SetTextLabel(LabelText);
	}
}

bool ADebugMenuLibraryObjectSearch::StringMatchesSearch(const FString& CheckString, TArray<FString>& SearchWords, bool bUseExactMatch) const
{
	for (FString& SearchWord : SearchWords)
	{
		if (bUseExactMatch)
		{
			if (CheckString.Compare(SearchWord, ESearchCase::IgnoreCase) != 0)
			{
				return false;
			}
		}
		else
		{
			if (!CheckString.Contains(SearchWord, ESearchCase::IgnoreCase))
			{
				return false;
			}
		}
	}

	return true;
}

void ADebugMenuLibraryObjectSearch::OnMouseClicked()
{
	if (!bMousePicker)
	{
		return;
	}

	UWorld* World = GetWorld();
	if (World == nullptr)
	{
		return;
	}

	APlayerController* PlayerController = World->GetFirstPlayerController();
	if (PlayerController == nullptr)
	{
		return;
	}

	FVector WorldLocation;
	FVector WorldDirection;
	if (!PlayerController->DeprojectMousePositionToWorld(WorldLocation,  WorldDirection))
	{
		return;
	}

	TArray<FHitResult> Hits;
	FCollisionQueryParams Params;
	const float TraceDist = 50000.0f;
	
	if (!GetWorld()->LineTraceMultiByChannel(Hits, WorldLocation, WorldLocation + (WorldDirection * TraceDist), ECC_Pawn, Params))
	{
		return;
	}

	FHitResult HitResult;
	for (const auto& Hit : Hits)
	{
		if (Hit.IsValidBlockingHit())
		{
			HitResult = Hit;
			break;
		}
	}

	AActor* BestHitActor = HitResult.GetActor();
	if (!BestHitActor)
	{
		return;
	}

	SearchText = BestHitActor->GetName();
	PerformSearch(BestHitActor);

	ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(this);
	if (MenuManager)
	{
		MenuManager->RefreshMenu();
	}
}

void ADebugMenuLibraryObjectSearch::BeginPlay()
{
	ObjSearchInputComponent = NewObject<UInputComponent>(this, TEXT("Input"));
	if (ObjSearchInputComponent)
	{
		ObjSearchInputComponent->RegisterComponent();
		ObjSearchInputComponent->bBlockInput = false;
		ObjSearchInputComponent->Priority = 999;

		ObjSearchInputComponent->BindKey(EKeys::LeftMouseButton, IE_Pressed, this, &ADebugMenuLibraryObjectSearch::OnMouseClicked);
	}

	MenuEntries.Empty();

	FString PropertyName = "SearchText";
	FProperty* FoundProperty = FindFProperty<FStrProperty>(this->GetClass(), *PropertyName);
	if (FoundProperty)
	{
		FName ObjName;
		UDebugMenuEntryStringUObjectProp* NewProp = NewObject<UDebugMenuEntryStringUObjectProp>(this, ObjName);
		if (NewProp)
		{
			NewProp->TargetObject = TWeakObjectPtr<UObject>(this);
			NewProp->TargetProperty = FoundProperty;
			NewProp->OnStringCommitted.AddDynamic(this, &ADebugMenuLibraryObjectSearch::OnSearchFilterStringUpdated);
			MenuEntries.Add(NewProp);
		}
	}

	PropertyName = "MaxSearchResults";
	FoundProperty = FindFProperty<FIntProperty>(this->GetClass(), *PropertyName);
	if (FoundProperty)
	{
		FName ObjName;
		UDebugMenuEntryIntUObjectProp* NewProp = NewObject<UDebugMenuEntryIntUObjectProp>(this, ObjName);
		if (NewProp)
		{
			NewProp->TargetObject = TWeakObjectPtr<UObject>(this);
			NewProp->TargetProperty = FoundProperty;
			NewProp->SetTweakInterval(5);
			NewProp->OnValueChanged.AddDynamic(this, &ADebugMenuLibraryObjectSearch::OnSearchFilterIntegerUpdated);
			MenuEntries.Add(NewProp);
		}
	}

	PropertyName = "bSearchClassName";
	FoundProperty = FindFProperty<FBoolProperty>(this->GetClass(), *PropertyName);
	if (FoundProperty)
	{
		FName ObjName;
		UDebugMenuEntryBoolUObjectProp* NewProp = NewObject<UDebugMenuEntryBoolUObjectProp>(this, ObjName);
		if (NewProp)
		{
			NewProp->TargetObject = TWeakObjectPtr<UObject>(this);
			NewProp->TargetProperty = FoundProperty;
			NewProp->OnValueChanged.AddDynamic(this, &ADebugMenuLibraryObjectSearch::OnSearchFilterBoolUpdated);
			MenuEntries.Add(NewProp);
		}
	}

	PropertyName = "bMousePicker";
	FoundProperty = FindFProperty<FBoolProperty>(this->GetClass(), *PropertyName);
	if (FoundProperty)
	{
		FName ObjName;
		UDebugMenuEntryBoolUObjectProp* NewProp = NewObject<UDebugMenuEntryBoolUObjectProp>(this, ObjName);
		if (NewProp)
		{
			NewProp->TargetObject = TWeakObjectPtr<UObject>(this);
			NewProp->TargetProperty = FoundProperty;
			MenuEntries.Add(NewProp);
		}
	}

	PropertyName = "bExactMatch";
	FoundProperty = FindFProperty<FBoolProperty>(this->GetClass(), *PropertyName);
	if (FoundProperty)
	{
		FName ObjName;
		UDebugMenuEntryBoolUObjectProp* NewProp = NewObject<UDebugMenuEntryBoolUObjectProp>(this, ObjName);
		if (NewProp)
		{
			NewProp->TargetObject = TWeakObjectPtr<UObject>(this);
			NewProp->TargetProperty = FoundProperty;
			NewProp->OnValueChanged.AddDynamic(this, &ADebugMenuLibraryObjectSearch::OnSearchFilterBoolUpdated);
			MenuEntries.Add(NewProp);
		}
	}

	FName ClearWatchObjName;
	UDebugMenuEntryBlueprint* BlueprintEntry = NewObject<UDebugMenuEntryBlueprint>(this, ClearWatchObjName);
	if (BlueprintEntry)
	{
		BlueprintEntry->DefaultDisplayString = "Clear Watches";
		BlueprintEntry->FunctionName_OnExecute = FName(TEXT("OnClearWatches"));
		MenuEntries.Add(BlueprintEntry);
	}
	
	Super::BeginPlay();
}

void ADebugMenuLibraryObjectSearch::EndPlay(const EEndPlayReason::Type EndPlayReason)
{
	ObjSearchInputComponent = nullptr;
	MenuEntries.Empty();

	Super::EndPlay(EndPlayReason);
}

void ADebugMenuLibraryObjectSearch::OnLibraryDisplayed()
{		
	Super::OnLibraryDisplayed();

	if (UWorld* World = GetWorld())
	{
		APlayerController* PlayerController = World->GetFirstPlayerController();
		if (PlayerController && ObjSearchInputComponent)
		{
			PlayerController->PushInputComponent(ObjSearchInputComponent);
		}
	}
}

void ADebugMenuLibraryObjectSearch::OnLibraryExited()
{
	if (UWorld* World = GetWorld())
	{
		APlayerController* PlayerController = World->GetFirstPlayerController();
		if (PlayerController && ObjSearchInputComponent)
		{
			PlayerController->PopInputComponent(ObjSearchInputComponent);
		}
	}

	Super::OnLibraryExited();
}

void ADebugMenuLibraryObjectSearch::OnClearWatches()
{
	if (UWorld* World = GetWorld())
	{
		ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(World);
		if (MenuManager)
		{
			MenuManager->ClearObjectPropertyWatches();
		}
	}
}