// Copyright (c) 2022 FWORKS. All Rights Reserved.

#include "DebugMenuManager.h"
#include "DebugMenuLibrary.h"
#include "DebugMenuWorldSubsystem.h"
#include "DebugMenuEntry.h"
#include "HAL/IConsoleManager.h"
#include "GameFramework/PlayerController.h"
#include "GameFramework/Pawn.h"
#include "EngineUtils.h"
#include "Engine.h"
#include "GameplayTagContainer.h"

TAutoConsoleVariable<float> CVarDebugMenuWidthMultiplier(
	TEXT("f.DebugMenu.WidthMultiplier"),
	1.0f,
	TEXT("Expand the width of debug menu by this multiplier"),
	ECVF_Default);

FAutoConsoleVariableSink CVarShowDebugMenuSink(FConsoleCommandDelegate::CreateStatic(&ADebugMenuManager::DebugMenuSink));

void ADebugMenuManager::DebugMenuSink()
{
	static float PreviousWidthMultiplier = 1.0f;
	float NewWidthMultiplier = CVarDebugMenuWidthMultiplier.GetValueOnGameThread();
	if (NewWidthMultiplier != PreviousWidthMultiplier)
	{
		PreviousWidthMultiplier = NewWidthMultiplier;
		UWorld* World = GEngine->GameViewport->GetWorld();
		if (World)
		{
			ADebugMenuManager* MenuManager = GetDebugMenuManager(World);
			if (MenuManager)
			{
				MenuManager->RecreateMenu();
			}
		}
	}
}

ADebugMenuManager::ADebugMenuManager()
{
	PrimaryActorTick.bCanEverTick = true;
	PrimaryActorTick.bStartWithTickEnabled = false;
}

void ADebugMenuManager::BeginPlay()
{
	Super::BeginPlay();

	InitializeMenuManager();

	FActorSpawnParameters SpawnParams;
	SpawnParams.Owner = this;
	for (TSubclassOf<ADebugMenuLibrary> LibraryClass : DefaultLibraries)
	{
		if (LibraryClass)
		{
			ADebugMenuLibrary* SpawnedLibrary = GetWorld()->SpawnActor<ADebugMenuLibrary>(LibraryClass, SpawnParams);
			if (SpawnedLibrary)
			{
				SpawnedLibraries.Add(SpawnedLibrary);
			}
		}
	}
}

ADebugMenuManager* ADebugMenuManager::GetDebugMenuManager(const UObject* WorldContextObj)
{
	if (WorldContextObj == nullptr || WorldContextObj->GetWorld() == nullptr)
	{
		return nullptr;
	}

	UDebugMenuWorldSubsystem* MenuSubsystem = UWorld::GetSubsystem<UDebugMenuWorldSubsystem>(WorldContextObj->GetWorld());
	if (MenuSubsystem)
	{
		return MenuSubsystem->GetManager();
	}

	return nullptr;
}

void ADebugMenuManager::InitializeMenuManager()
{
	if (bHasInitialized)
	{
		return;
	}

	// add root node to hierarchy
	DebugMenuHierarchy.Empty();
	FDebugMenuHierarchyNode RootNode;
	DebugMenuHierarchy.Add(RootNode);

	bHasInitialized = true;
}

void ADebugMenuManager::EndPlay(const EEndPlayReason::Type EndPlayReason)
{
	if (DebugMenuHierarchy.Num() > 0)
	{
		for (int32 i = 0; i < DebugMenuHierarchy[CurrentHierarchyIndex].Libraries.Num(); i++)
		{
			ADebugMenuLibrary* LibraryDef = DebugMenuHierarchy[CurrentHierarchyIndex].Libraries[i].Get();
			if (LibraryDef == nullptr)
			{
				continue;
			}

			LibraryDef->OnLibraryExited();
		}
	}

	if (DebugMenuWidget.IsValid())
	{
		if (GEngine->GameViewport)
		{
			GEngine->GameViewport->RemoveViewportWidgetContent(DebugMenuWidget.ToSharedRef());
		}
		
		DebugMenuWidget.Reset();
	}

	for (ADebugMenuLibrary* SpawnedLibrary : SpawnedLibraries)
	{
		if (SpawnedLibrary != nullptr)
		{
			SpawnedLibrary->Destroy();
			SpawnedLibrary = nullptr;
		}
	}
	SpawnedLibraries.Empty();

	Super::EndPlay(EndPlayReason);
}

void ADebugMenuManager::InputDown()
{
	if (DebugMenuWidget.IsValid())
	{
		DebugMenuWidget->SelectDown();
	}
}

void ADebugMenuManager::InputUp()
{
	if (DebugMenuWidget.IsValid())
	{
		DebugMenuWidget->SelectUp();
	}
}

void ADebugMenuManager::InputLeft()
{
	if (DebugMenuWidget.IsValid())
	{
		TSharedPtr<FDebugMenuListEntryData> CurrSelected = DebugMenuWidget->GetSelectedEntry();
		if (CurrSelected.IsValid())
		{
			if (UDebugMenuEntry* EntryData = CurrSelected->EntryData.Get())
			{
				bool bRefresh = EntryData->OnInputLeft(CurrSelected, this);
				if (bRefresh)
				{
					RefreshMenu();
				}
			}
		}
	}
}

void ADebugMenuManager::InputRight()
{
	if (DebugMenuWidget.IsValid())
	{
		TSharedPtr<FDebugMenuListEntryData> CurrSelected = DebugMenuWidget->GetSelectedEntry();
		if (CurrSelected.IsValid())
		{
			if (UDebugMenuEntry* EntryData = CurrSelected->EntryData.Get())
			{
				bool bRefresh = EntryData->OnInputRight(CurrSelected, this);
				if (bRefresh)
				{
					RefreshMenu();
				}
			}
		}
	}
}

bool ADebugMenuManager::IsPropertyBeingWatched(FDebugMenuObjPropWatch& CheckProperty)
{
	for (FDebugMenuObjPropWatch& CurrWatch : PropertyWatches)
	{
		if (CurrWatch.TargetObject != CheckProperty.TargetObject)
		{
			continue;
		}

		if (CurrWatch.TargetProperty != CheckProperty.TargetProperty)
		{
			continue;
		}

		if (CurrWatch.PropertyChain.Num() != CheckProperty.PropertyChain.Num())
		{
			continue;
		}

		bool bChainPass = true;
		for (int32 i = 0; i < CurrWatch.PropertyChain.Num(); i++)
		{
			if (CurrWatch.PropertyChain[i].ArrayIndex != CheckProperty.PropertyChain[i].ArrayIndex)
			{
				bChainPass = false;
				break;
			}

			if (CurrWatch.PropertyChain[i].Property != CheckProperty.PropertyChain[i].Property)
			{
				bChainPass = false;
				break;
			}
		}

		if (!bChainPass)
		{
			continue;
		}

		return true;
	}

	return false;
}

void ADebugMenuManager::ClearObjectPropertyWatches()
{
	PropertyWatches.Empty();
}

void ADebugMenuManager::InputToggleWatch()
{
	if (DebugMenuWidget.IsValid())
	{
		TSharedPtr<FDebugMenuListEntryData> CurrSelected = DebugMenuWidget->GetSelectedEntry();
		if (CurrSelected.IsValid())
		{
			if (UDebugMenuEntry* EntryData = CurrSelected->EntryData.Get())
			{
				// adding/removing a watch
				 EntryData->OnInputToggleWatch(CurrSelected, this);
				 DebugMenuWidget->RefreshEntries();
			}
		}
	}
}

void ADebugMenuManager::ToggleObjectPropertyWatch(FDebugMenuObjPropWatch WatchData)
{
	bool bFound = false;
	for (int32 i = 0; i < PropertyWatches.Num(); i++)
	{
		if (PropertyWatches[i].TargetObject != WatchData.TargetObject)
		{
			continue;
		}

		if (PropertyWatches[i].TargetProperty != WatchData.TargetProperty)
		{
			continue;
		}

		if (PropertyWatches[i].PropertyChain.Num() != WatchData.PropertyChain.Num())
		{
			continue;
		}

		bool bChainPass = true;
		for (int32 j = 0; j < PropertyWatches[i].PropertyChain.Num(); i++)
		{
			if (PropertyWatches[i].PropertyChain[j].ArrayIndex != WatchData.PropertyChain[j].ArrayIndex)
			{
				bChainPass = false;
				break;
			}

			if (PropertyWatches[i].PropertyChain[j].Property != WatchData.PropertyChain[j].Property)
			{
				bChainPass = false;
				break;
			}
		}

		if (!bChainPass)
		{
			continue;
		}

		bFound = true;
		PropertyWatches.RemoveAtSwap(i);
		break;
	}

	if (!bFound)
	{
		PropertyWatches.Add(WatchData);
	}

	if (PropertyWatches.Num() > 0)
	{
		SetActorTickEnabled(true);
	}
	else
	{
		SetActorTickEnabled(false);
	}
}

void ADebugMenuManager::Tick(float DeltaSeconds)
{
	Super::Tick(DeltaSeconds);

	for (int32 i = PropertyWatches.Num() - 1; i >= 0; i--)
	{
		UObject* TargetObjectRaw = PropertyWatches[i].TargetObject.Get();
		FProperty* TargetPropertyRaw = PropertyWatches[i].TargetProperty.Get();
		if (!TargetObjectRaw)
		{
			PropertyWatches.RemoveAtSwap(i);
			continue;
		}

		if (!TargetPropertyRaw)
		{
			PropertyWatches.RemoveAtSwap(i);
			continue;
		}

		FString ObjPrefix = TargetObjectRaw->GetName();
		if (AActor* ActorOuter = TargetObjectRaw->GetTypedOuter<AActor>())
		{
			if (!TargetObjectRaw->IsA<AActor>())
			{
				ObjPrefix = ActorOuter->GetName() + "(" + ObjPrefix + ")";
			}
		}
		ObjPrefix += " - ";

		FString WatchString = ObjPrefix + PropertyWatches[i].DisplayPrefix + GetObjectPropertyValueAsString(TargetObjectRaw, TargetPropertyRaw, TargetPropertyRaw->GetName(), PropertyWatches[i].PropertyChain);
		GEngine->AddOnScreenDebugMessage(-1, 0.0f, FColor::Yellow, WatchString);
	}

	if (PropertyWatches.Num() <= 0)
	{
		SetActorTickEnabled(false);
	}
}

void ADebugMenuManager::InputConfirm()
{
	if (DebugMenuWidget.IsValid())
	{
		TSharedPtr<FDebugMenuListEntryData> CurrSelected = DebugMenuWidget->GetSelectedEntry();
		if (CurrSelected.IsValid())
		{
			ADebugMenuLibrary* LibData = CurrSelected->LibrariesData.Num() > 0 ? CurrSelected->LibrariesData[0].Get() : nullptr;
			if (LibData)
			{
				// selecting a subcategory
				int32 CurrIndex = GetCurrentHierarchyDisplayIndex();
				for (int32 i = 0; i < DebugMenuHierarchy[CurrIndex].ChildNodes.Num(); i++)
				{
					int32 ChildIndex = DebugMenuHierarchy[CurrIndex].ChildNodes[i];
					if (DebugMenuHierarchy[ChildIndex].CategoryName == LibData->GetCategoryDisplayFName())
					{
						PushSelectionCrumb();
						SetCurrentHierarchyDisplayIndex(ChildIndex);
						break;
					}
				}

				DebugMenuWidget->RefreshEntries();
			}
			else if (UDebugMenuEntry* EntryData = CurrSelected->EntryData.Get())
			{
				// selecting an entry
				bool bRefresh = EntryData->OnInputConfirm(CurrSelected, this);
				if (bRefresh)
				{
					RefreshMenu();
				}
			}
			else if (CurrSelected->bIsGoBack)
			{
				InputCancel();
			}
		}
	}
}

void ADebugMenuManager::RefreshMenu()
{
	if (DebugMenuWidget.IsValid())
	{
		DebugMenuWidget->RefreshEntries();
	}
}

void ADebugMenuManager::InputCancel()
{
	int32 CurrIndex = GetCurrentHierarchyDisplayIndex();
	if (CurrIndex == 0)
	{
		ToggleDebugMenu();
		return;
	}

	if (DebugMenuWidget.IsValid())
	{
		TSharedPtr<FDebugMenuListEntryData> CurrSelected = DebugMenuWidget->GetSelectedEntry();
		if (CurrSelected.IsValid())
		{
			if (CurrSelected->EntryData.IsValid() && CurrSelected->EntryData->OnInputCancel(CurrSelected, this))
			{
				// menu entry has consumed the cancel button
				return;
			}
		}
	}

	const int32 OldIndex = CurrentHierarchyIndex;
	
	SetCurrentHierarchyDisplayIndex(DebugMenuHierarchy[CurrentHierarchyIndex].ParentNode);

	// inform the category that we're exiting
	for (int32 i = 0; i < DebugMenuHierarchy[OldIndex].Libraries.Num(); i++)
	{
		ADebugMenuLibrary* LibraryDef = DebugMenuHierarchy[OldIndex].Libraries[i].Get();
		if (LibraryDef == nullptr)
		{
			continue;
		}

		LibraryDef->OnLibraryExited();
	}

	if (DebugMenuWidget.IsValid())
	{
		DebugMenuWidget->RefreshEntries(true);
	}

	PopSelectionCrumb();
}

bool ADebugMenuManager::IsWidgetInChildHierarchy(TSharedPtr<SWidget> Child, TSharedPtr<SWidget> Parent) const
{
	SWidget* Current = Child.Get();
	while ((Current!= nullptr) && (Current->GetParentWidget().IsValid()))
	{
		if (Current->GetParentWidget() == Parent)
		{
			return true;
		}

		Current = Current->GetParentWidget().Get();
	}

	return false;
}

bool ADebugMenuManager::IsDebugMenuActive() const
{
	return DebugMenuWidget.IsValid() && (DebugMenuWidget->GetVisibility() == EVisibility::Visible);
}

void ADebugMenuManager::RecreateMenu()
{
	if (DebugMenuWidget.IsValid())
	{
		HideDebugMenu();

		if (GEngine->GameViewport)
		{
			GEngine->GameViewport->RemoveViewportWidgetContent(DebugMenuWidget.ToSharedRef());
		}

		DebugMenuWidget.Reset();
	}

	ShowDebugMenu();
}

void ADebugMenuManager::ShowDebugMenu()
{
	if (!DebugMenuWidget.IsValid())
	{
		DebugMenuWidget = SNew(SDebugMenuWidget)
			.bRightSide(bDisplayOnRightSide)
			.MenuWorld(TWeakObjectPtr<UWorld>(GetWorld()))
			.MenuBGColor(MenuBGColor)
			.MenuHighlightedBGColor(MenuHighlightedBGColor)
			.MenuRowMargin(MenuRowMargin)
			.MenuWidth(MenuWidth * CVarDebugMenuWidthMultiplier.GetValueOnAnyThread())
			.EntryFont(EntryFont)
			.CategoryFont(CategoryFont);

		DebugMenuWidget->AssociatedMenuManager = TWeakObjectPtr<ADebugMenuManager>(this);
		DebugMenuWidget->RefreshEntries();

		// todo: should we pass a weakpointer instead? to not give viewport ownership of the widget
		GEngine->GameViewport->AddViewportWidgetContent(
			DebugMenuWidget.ToSharedRef(),
			99 // Using a high Z order to always render it on top of any widgets we may have on the screen.
		);

		OnDebugMenuShown.Broadcast();
	}
	else
	{
		if (DebugMenuWidget->GetVisibility() == EVisibility::Collapsed)
		{
			DebugMenuWidget->RefreshEntries();
			DebugMenuWidget->SetVisibility(EVisibility::Visible);

			OnDebugMenuShown.Broadcast();
		}
	}
}

void ADebugMenuManager::HideDebugMenu()
{
	if (DebugMenuWidget.IsValid() && DebugMenuWidget->GetVisibility() == EVisibility::Visible)
	{
		DebugMenuWidget->RequestHideMenu();

		OnDebugMenuHidden.Broadcast();
	}
}

void ADebugMenuManager::ToggleDebugMenu()
{
	if (IsDebugMenuActive())
	{
		HideDebugMenu();
	}
	else
	{
		ShowDebugMenu();
	}
}

int32 ADebugMenuManager::AddChildToHierarchy(FName ChildName, int32 ParentId)
{
	if (ParentId == -1)
	{
		return -1;
	}

	if (ParentId >= DebugMenuHierarchy.Num())
	{
		return -1;
	}

	// look for an unused slot first
	for (int32 i = 1; i < DebugMenuHierarchy.Num(); i++)
	{
		if (DebugMenuHierarchy[i].ParentNode == -1)
		{
			DebugMenuHierarchy[ParentId].ChildNodes.Add(i);
			DebugMenuHierarchy[i].ParentNode = ParentId;
			DebugMenuHierarchy[i].CategoryName = ChildName;
			return i;
		}
	}

	int32 NewIndex = DebugMenuHierarchy.Num();
	FDebugMenuHierarchyNode NewNode;
	NewNode.CategoryName = ChildName;
	NewNode.ParentNode = ParentId;
	DebugMenuHierarchy[ParentId].ChildNodes.Add(NewIndex);
	DebugMenuHierarchy.Add(NewNode);
	return NewIndex;
}

void ADebugMenuManager::RegisterLibrary(ADebugMenuLibrary* Library)
{
	if (Library == nullptr)
	{
		return;
	}

	if (!bHasInitialized)
	{
		InitializeMenuManager();
	}

	if (DebugMenuHierarchy.Num() < 1)
	{
		return;
	}

	TWeakObjectPtr<ADebugMenuLibrary> LibraryWPtr = TWeakObjectPtr<ADebugMenuLibrary>(Library);
	TArray<FName> CategoryHierarchy = Library->GetCategoryHierarchy();
	if (CategoryHierarchy.Num() == 0)
	{
		DebugMenuHierarchy[0].Libraries.Add(LibraryWPtr);
		return;
	}

	// for each level in this categorydef's hierarchy
	int32 CurrentIndexInHierarchy = 0;
	for (int32 i = 0; i < CategoryHierarchy.Num(); i++)
	{
		// see if we can find a matching node in the menu's hiearchy
		bool bFound = false;
		for (int32 j = 0; j < DebugMenuHierarchy[CurrentIndexInHierarchy].ChildNodes.Num(); j++)
		{
			int32 ChildIndex = DebugMenuHierarchy[CurrentIndexInHierarchy].ChildNodes[j];
			if (DebugMenuHierarchy[ChildIndex].CategoryName == CategoryHierarchy[i])
			{
				bFound = true;
				CurrentIndexInHierarchy = ChildIndex;
				break;
			}
		}

		if (!bFound)
		{
			CurrentIndexInHierarchy = AddChildToHierarchy(CategoryHierarchy[i], CurrentIndexInHierarchy);
			if (CurrentIndexInHierarchy < 0)
			{
				// something went wrong
				return;
			}
		}
	}

	DebugMenuHierarchy[CurrentIndexInHierarchy].Libraries.Add(LibraryWPtr);
}

void ADebugMenuManager::UnregisterLibrary(ADebugMenuLibrary* Library)
{
	if (Library == nullptr)
	{
		return;
	}

	if (!bHasInitialized)
	{
		InitializeMenuManager();
	}

	if (DebugMenuHierarchy.Num() < 1)
	{
		return;
	}

	TWeakObjectPtr<ADebugMenuLibrary> LibraryWPtr = TWeakObjectPtr<ADebugMenuLibrary>(Library);
	TArray<FName> CategoryHierarchy = Library->GetCategoryHierarchy();
	if (CategoryHierarchy.Num() == 0)
	{
		DebugMenuHierarchy[0].Libraries.Remove(LibraryWPtr);
		return;
	}

	// for each level in this categorydef's hierarchy
	int32 CurrentIndexInHierarchy = 0;
	for (int32 i = 0; i < CategoryHierarchy.Num(); i++)
	{
		// see if we can find a matching node in the menu's hiearchy
		bool bFound = false;
		for (int32 j = 0; j < DebugMenuHierarchy[CurrentIndexInHierarchy].ChildNodes.Num(); j++)
		{
			int32 ChildIndex = DebugMenuHierarchy[CurrentIndexInHierarchy].ChildNodes[j];
			if (DebugMenuHierarchy[ChildIndex].CategoryName == CategoryHierarchy[i])
			{
				bFound = true;
				CurrentIndexInHierarchy = ChildIndex;
				break;
			}
		}

		if (!bFound)
		{
			// something went wrong, we couldn't find the path?
			UE_LOG(LogTemp, Log, TEXT("Tried to unregister library but could not find it in menu hierarchy"));
			return;
		}
	}

	DebugMenuHierarchy[CurrentIndexInHierarchy].Libraries.Remove(LibraryWPtr);

	//todo: we may want to clear the node if it is no longer used? leave it for now
	if (DebugMenuHierarchy[CurrentIndexInHierarchy].Libraries.Num() <= 0
		&& DebugMenuHierarchy[CurrentIndexInHierarchy].ChildNodes.Num() <= 0)
	{
		int32 ParentNode = DebugMenuHierarchy[CurrentIndexInHierarchy].ParentNode;
		if (ParentNode >= 0)
		{
			DebugMenuHierarchy[ParentNode].ChildNodes.Remove(CurrentIndexInHierarchy);
			DebugMenuHierarchy[CurrentIndexInHierarchy].ParentNode = -1;
		}
	}
	
}

int32 ADebugMenuManager::GetCurrentHierarchyDisplayIndex() const
{
	if (DebugMenuHierarchy.Num() <= 0)
	{
		return -1;
	}

	return CurrentHierarchyIndex;
}

void ADebugMenuManager::PushSelectionCrumb()
{
	// push a selection crumb
	if (DebugMenuWidget.IsValid())
	{
		int32 SelectedIndex = DebugMenuWidget->GetSelectedIndex();
		SelectionCrumbs.Add(SelectedIndex);
	}
}

void ADebugMenuManager::PopSelectionCrumb()
{
	if (SelectionCrumbs.Num() > 0)
	{
		SelectionCrumbs.RemoveAt(SelectionCrumbs.Num() - 1);
	}
}

void ADebugMenuManager::SetCurrentHierarchyDisplayIndex(int32 NewIndex)
{
	CurrentHierarchyIndex = NewIndex;
}

int32 ADebugMenuManager::GetSelectionCrumb() const
{
	if (SelectionCrumbs.Num() > 0)
	{
		return SelectionCrumbs[SelectionCrumbs.Num() - 1];
	}

	return 0;
}

int32 ADebugMenuManager::FindHierarchyIndex(TArray<FName> InCategoryHierarchy) const
{
	int32 CurrentNameIndex = 0;
	int32 TempHierarchyIndex = 0;
	if (DebugMenuHierarchy.Num() <= 0)
	{
		return -1;
	}

	bool bFound = true;
	while (bFound)
	{
		bFound = false;
		for (int32 Child : DebugMenuHierarchy[TempHierarchyIndex].ChildNodes)
		{
			if (Child >= DebugMenuHierarchy.Num())
			{
				continue;
			}

			if (DebugMenuHierarchy[Child].CategoryName == InCategoryHierarchy[CurrentNameIndex])
			{
				CurrentNameIndex++;
				if (CurrentNameIndex >= InCategoryHierarchy.Num())
				{
					return Child;
				}

				// go search the next level
				TempHierarchyIndex = Child;
				bFound = true;
				break;
			}
		}
	}
	
	return -1;
}

void ADebugMenuManager::OnCategoryEntriesDisplayed()
{
	TArray< TWeakObjectPtr<UDebugMenuEntry> > Entries;

	if (DebugMenuHierarchy.Num() <= 0)
	{
		return;
	}

	int32 Curr = GetCurrentHierarchyDisplayIndex();
	for (int32 i = 0; i < DebugMenuHierarchy[Curr].Libraries.Num(); i++)
	{
		ADebugMenuLibrary* LibraryDef = DebugMenuHierarchy[Curr].Libraries[i].Get();
		if (LibraryDef == nullptr)
		{
			continue;
		}

		LibraryDef->OnLibraryDisplayed();
	}
}

TArray< TWeakObjectPtr<UDebugMenuEntry> > ADebugMenuManager::GetCurrentDisplayedEntries() const
{
	TArray< TWeakObjectPtr<UDebugMenuEntry> > Entries;

	if (DebugMenuHierarchy.Num() <= 0)
	{
		return Entries;
	}

	int32 Curr = GetCurrentHierarchyDisplayIndex();
	for (int32 i = 0; i < DebugMenuHierarchy[Curr].Libraries.Num(); i++)
	{
		ADebugMenuLibrary* LibraryDef = DebugMenuHierarchy[Curr].Libraries[i].Get();
		if (LibraryDef == nullptr)
		{
			continue;
		}

		for (int32 j = 0; j < LibraryDef->GetEntries().Num(); j++)
		{
			if (LibraryDef->GetEntries()[j] == nullptr)
			{
				continue;
			}

			Entries.Add(TWeakObjectPtr<UDebugMenuEntry>(LibraryDef->GetEntries()[j]));
		}
	}

	return Entries;
}

void* ADebugMenuManager::GetContainerPtrFromPropertyChain(const UObject* InObject, TArray<FDebugMenuPropertyChain> PropertyChain)
{
	if (InObject == nullptr)
	{
		return nullptr;
	}

	void* ContainerAddress = (void*)InObject;

	for (int32 i = 0; i < PropertyChain.Num(); i++)
	{
		FProperty* CurrProperty = PropertyChain[i].Property.Get();
		if (CurrProperty == nullptr)
		{
			continue;
		}

		if (FStructProperty* PropStruct = CastField<FStructProperty>(CurrProperty))
		{
			ContainerAddress = CurrProperty->ContainerPtrToValuePtr<void>(ContainerAddress);
		}
		else if (FArrayProperty* ArrayProperty = CastField<FArrayProperty>(CurrProperty))
		{
			void* ArrayContainerAddress = ArrayProperty->ContainerPtrToValuePtr<void>(ContainerAddress);

			FScriptArrayHelper Helper(ArrayProperty, ArrayContainerAddress);
			const int32 NumElements = Helper.Num();
			const int32 ArrayIndex = PropertyChain[i].ArrayIndex;
			if (ArrayIndex >= 0
				&& ArrayIndex < NumElements
				&& Helper.IsValidIndex(ArrayIndex))
			{
				ContainerAddress = Helper.GetRawPtr(ArrayIndex);
			}
		}
		if (const FMapProperty* MapProperty = CastField<FMapProperty>(CurrProperty))
		{
			const FProperty* ValueProp = MapProperty->GetValueProperty();
			const FProperty* KeyProp = MapProperty->GetKeyProperty();
			void* MapContainerAddress = MapProperty->ContainerPtrToValuePtr<void>(ContainerAddress);
			FScriptMapHelper MapHelper(MapProperty, MapContainerAddress);
			const int32 ArrayIndex = PropertyChain[i].ArrayIndex;
			if (PropertyChain[i].bMapKey)
			{
				ContainerAddress = (void*)MapHelper.GetKeyPtr(ArrayIndex);
			}
			else
			{
				ContainerAddress = (void*)MapHelper.GetValuePtr(ArrayIndex);
			}
		}
	}

	return ContainerAddress;
}


FString ADebugMenuManager::GetObjectPropertyValueAsString(const UObject* InObject, FProperty* Property, FString PropertyNameOverride, TArray<FDebugMenuPropertyChain> PropertyChain)
{
	FString Result = "";
	if (InObject == nullptr
		|| InObject->GetClass() == nullptr
		|| Property == nullptr)
	{
		return Result;
	}

	FString PropertyName = (PropertyNameOverride.Len() > 0) ? PropertyNameOverride : Property->GetName();
	void* ContainerAddress = ADebugMenuManager::GetContainerPtrFromPropertyChain(InObject, PropertyChain);

	if (FDoubleProperty* PropDouble = CastField<FDoubleProperty>(Property))
	{
		const double PropVal = PropDouble->GetPropertyValue_InContainer(ContainerAddress);
		Result += PropertyName + ": " + FString::SanitizeFloat(PropVal);
	}
	else if (FFloatProperty* PropFloat = CastField<FFloatProperty>(Property))
	{
		const float PropVal = PropFloat->GetPropertyValue_InContainer(ContainerAddress);
		Result += PropertyName + ": " + FString::SanitizeFloat(PropVal);
	}
	else if (FByteProperty* PropByte = CastField<FByteProperty>(Property))
	{
#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 1
		const int64 EnumValue = PropByte->GetSignedIntPropertyValue_InContainer(ContainerAddress);
#else
		const int64 EnumValue = PropByte->GetPropertyValue_InContainer(ContainerAddress);
#endif
		const UEnum* Enum = PropByte->Enum;
		if (Enum && Enum->IsValidEnumValue(EnumValue))
		{
			FString EnumStr = Enum->GetDisplayNameTextByValue(EnumValue).ToString();
			Result += PropertyName + ": " + EnumStr;
		}
		else
		{
			Result += PropertyName + ": " + FString::FromInt(EnumValue);
		}
	}
	else if (FBoolProperty* PropBool = CastField<FBoolProperty>(Property))
	{
		const bool PropVal = PropBool->GetPropertyValue_InContainer(ContainerAddress);
		Result += PropertyName + ": " + (PropVal ? "TRUE" : "FALSE");
	}
	else if (FIntProperty* PropInt = CastField<FIntProperty>(Property))
	{
		const int32 PropVal = PropInt->GetPropertyValue_InContainer(ContainerAddress);
		Result += PropertyName + ": " + FString::FromInt(PropVal);
	}
	else if (FStrProperty* PropStr = CastField<FStrProperty>(Property))
	{
		const FString PropVal = PropStr->GetPropertyValue_InContainer(ContainerAddress);
		Result += PropertyName + ": " + PropVal;
	}
	else if (FNameProperty* PropName = CastField<FNameProperty>(Property))
	{
		const FName PropVal = PropName->GetPropertyValue_InContainer(ContainerAddress);
		Result += PropertyName + ": " + PropVal.ToString();
	}
	else if (FStructProperty* PropStruct = CastField<FStructProperty>(Property))
	{
		if (PropStruct->Struct)
		{
			if (PropStruct->Struct->GetName() == "Quat")
			{
				FQuat PropVal = FQuat::Identity;
#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 1
				PropStruct->GetValue_InContainer(ContainerAddress, &PropVal);
#elif ENGINE_MAJOR_VERSION == 5
				// Get a pointer to the struct instance
				PropVal = *PropStruct->ContainerPtrToValuePtr<FQuat>(ContainerAddress);
#endif

				FRotator PropValRot = FRotator(PropVal);

				Result += PropertyName + ": ("
					+ FString::SanitizeFloat(PropValRot.Pitch) + ", "
					+ FString::SanitizeFloat(PropValRot.Yaw) + ", "
					+ FString::SanitizeFloat(PropValRot.Roll) + ")";
			}
			else if (PropStruct->Struct->GetName() == "Vector")
			{
				FVector PropVal = FVector::ZeroVector;
#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 1
				PropStruct->GetValue_InContainer(ContainerAddress, &PropVal);

#elif ENGINE_MAJOR_VERSION == 5
				// Get a pointer to the struct instance
				PropVal = *PropStruct->ContainerPtrToValuePtr<FVector>(ContainerAddress);
#endif
				Result += PropertyName + ": ("
					+ FString::SanitizeFloat(PropVal.X) + ", "
					+ FString::SanitizeFloat(PropVal.Y) + ", "
					+ FString::SanitizeFloat(PropVal.Z) + ")";

			}
			else if (PropStruct->Struct->GetName() == "Rotator")
			{
				FRotator PropVal = FRotator::ZeroRotator;
#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 1
				PropStruct->GetValue_InContainer(ContainerAddress, &PropVal);
#elif ENGINE_MAJOR_VERSION == 5
				// Get a pointer to the struct instance
				PropVal = *PropStruct->ContainerPtrToValuePtr<FRotator>(ContainerAddress);
#endif
				Result += PropertyName + ": ("
					+ FString::SanitizeFloat(PropVal.Pitch) + ", "
					+ FString::SanitizeFloat(PropVal.Yaw) + ", "
					+ FString::SanitizeFloat(PropVal.Roll) + ")";
			}
			else if (PropStruct->Struct == FGameplayTagContainer::StaticStruct())
			{
				const FGameplayTagContainer TagContainer = *PropStruct->ContainerPtrToValuePtr<FGameplayTagContainer>(ContainerAddress);
				TArray<FGameplayTag> GameplayTags;
				TagContainer.GetGameplayTagArray(GameplayTags);

				Result += PropertyName;
				Result += ": ";
				for (FGameplayTag& Tag : GameplayTags)
				{
					Result += "\n       ";
					Result += Tag.ToString();
				}
			}
			else if (PropStruct->Struct == FGameplayTag::StaticStruct())
			{
				const FGameplayTag Tag = *PropStruct->ContainerPtrToValuePtr<FGameplayTag>(ContainerAddress);
				Result += PropertyName + ": " + Tag.ToString();
			}
			else
			{
				FDebugMenuPropertyChain NewEntry;
				NewEntry.Property = PropStruct;
				PropertyChain.Add(NewEntry);

				int32 BasePropertyChain = PropertyChain.Num();

				FField* Curr = PropStruct->Struct->ChildProperties;
				bool bFirstProp = true;
				while (Curr != nullptr)
				{
					FProperty* SubProp = CastField<FProperty>(Curr);
					if (SubProp)
					{
						FString NewPropName = PropertyName + "." + SubProp->GetName();
						FString Temp = GetObjectPropertyValueAsString(InObject, SubProp, NewPropName, PropertyChain);
						if (Temp.Len() > 0)
						{
							if (!bFirstProp || PropertyChain.Num() <= BasePropertyChain)
							{
								Result += "\n    ";
							}

							Result += Temp;
							bFirstProp = false;
						}

					}
					Curr = Curr->Next;
				}
			}
		}
	}
	else if (FObjectPropertyBase* PropObj = CastField<FObjectPropertyBase>(Property))
	{
		UObject* ObjectReferenced = PropObj->GetObjectPropertyValue_InContainer(ContainerAddress);
		if (ObjectReferenced && ObjectReferenced->IsValidLowLevelFast())
		{
			Result += PropertyName + ": " + ObjectReferenced->GetName();
		}
		else
		{
			Result += PropertyName + ": [Null]";
		}
	}
	else if (FEnumProperty* PropEnum = CastField<FEnumProperty>(Property))
	{
		void* ValuePtr = PropEnum->ContainerPtrToValuePtr<void>(ContainerAddress);
		FNumericProperty* UnderlyingProperty = PropEnum->GetUnderlyingProperty();
		int64 EnumValue = UnderlyingProperty->GetSignedIntPropertyValue(ValuePtr);
		FString ValueStr = "";
		UEnum* Enum = PropEnum->GetEnum();
		if (Enum)
		{
			if (Enum->IsValidEnumValue(EnumValue))
			{
				ValueStr = Enum->GetDisplayNameTextByValue(EnumValue).ToString();
			}
			else
			{
				ValueStr = "INVALID";
			}
		}
		else
		{
			ValueStr = FText::AsNumber(EnumValue).ToString();
		}

		Result += PropertyName + ": " + ValueStr;
	}
	else if (FArrayProperty* ArrayProperty = CastField<FArrayProperty>(Property))
	{
		void* ArrayContainerAddress = ArrayProperty->ContainerPtrToValuePtr<void>(ContainerAddress);

		FScriptArrayHelper Helper(ArrayProperty, ArrayContainerAddress);
		int32 NumElements = Helper.Num();
		Result += "Array: " + PropertyName + "(size " + FString::FromInt(NumElements) + ")";
		for (int32 i = 0; i < NumElements; i++)
		{
			if (Helper.IsValidIndex(i))
			{
				TArray<FDebugMenuPropertyChain> TempChain = PropertyChain;
				FDebugMenuPropertyChain ArrayEntry;
				ArrayEntry.ArrayIndex = i;
				ArrayEntry.Property = ArrayProperty;
				TempChain.Add(ArrayEntry);

				FString NewPropertyName = PropertyName + "[" + FString::FromInt(i) + "]";
				FString ArrayPropStr = GetObjectPropertyValueAsString(InObject, ArrayProperty->Inner, NewPropertyName, TempChain);
				if (ArrayPropStr.Len() > 0)
				{
					if (!ArrayPropStr.StartsWith("\n"))
					{
						Result += "\n";
					}
					Result += "    ";
					Result += ArrayPropStr;
				}
			}
		}
	}
	else if (FMapProperty* MapProperty = CastField<FMapProperty>(Property))
	{
		void* MapContainerAddress = MapProperty->ContainerPtrToValuePtr<void>(ContainerAddress);
		FScriptMapHelper MapHelper(MapProperty, MapContainerAddress);

		FProperty* ValueProp = MapHelper.GetValueProperty();
		FProperty* KeyProp = MapHelper.GetKeyProperty();

		for (auto Iter = MapHelper.CreateIterator(); Iter; ++Iter)
		{
			TArray<FDebugMenuPropertyChain> TempKeyChain = PropertyChain;
			FDebugMenuPropertyChain ArrayEntry;
			ArrayEntry.ArrayIndex = *Iter;
			ArrayEntry.Property = MapProperty;
			ArrayEntry.bMapKey = true;
			TempKeyChain.Add(ArrayEntry);

			TArray<FDebugMenuPropertyChain> TempValueChain = PropertyChain;
			FDebugMenuPropertyChain ArrayEntry2;
			ArrayEntry2.ArrayIndex = *Iter;
			ArrayEntry2.Property = MapProperty;
			ArrayEntry2.bMapKey = false;
			TempValueChain.Add(ArrayEntry);

			FString NewPropertyName = PropertyName + "[" + FString::FromInt(ArrayEntry.ArrayIndex) + "]";
			FString KeyString = GetObjectPropertyValueAsString(InObject, KeyProp, NewPropertyName, TempKeyChain);
			FString ValueString = GetObjectPropertyValueAsString(InObject, ValueProp, " ", TempValueChain);
			if (KeyString.Len() > 0)
			{
				if (!KeyString.StartsWith("\n"))
				{
					Result += "\n";
				}
				Result += "    ";
				Result += KeyString;
				Result += ValueString;
			}
		}
	}

	return Result;
}