// Copyright (c) 2022 FWORKS. All Rights Reserved.

#include "DebugMenuWidget.h"
#include "DebugMenuLibrary.h"
#include "HAL/IConsoleManager.h"
#include "GameFramework/PlayerController.h"
#include "DebugMenuManager.h"
#include "DebugMenuEntry.h"
#include "Framework/Application/SlateApplication.h"
#include "TimerManager.h"
#include "Engine/World.h"

#define LOCTEXT_NAMESPACE "DebugMenuWidget"

void SDebugMenuWidget::Construct(const FArguments& Args)
{
	AssociatedWorld = Args._MenuWorld;

	MenuBGColor = Args._MenuBGColor;
	MenuHighlightedBGColor = Args._MenuHighlightedBGColor;
	MenuRowMargin = Args._MenuRowMargin;
	MenuWidth = Args._MenuWidth;
	EntryFont = Args._EntryFont;
	CategoryFont = Args._CategoryFont;

	ListView =
		SNew(SListView<TSharedPtr<FDebugMenuListEntryData>>)
		.SelectionMode(ESelectionMode::Single)
		.ListItemsSource(&DisplayedEntries)
		.OnGenerateRow(this, &SDebugMenuWidget::OnGenerateRow)
		;

	FSlateBrush TransparentBrush;
	TransparentBrush.TintColor = FSlateColor(FLinearColor{ 0.0f, 0.0f, 0.0f, 0.0f });

	TransparentTableViewStyle = FAppStyle::Get().GetWidgetStyle<FTableViewStyle>("ListView");
	TransparentTableViewStyle.SetBackgroundBrush(TransparentBrush);
	ListView->SetStyle(&TransparentTableViewStyle);

	ListContainer = SNew(SBox)
		.WidthOverride(500)
		.Padding(FMargin(20.0f, 20.0f, 20.0f, 20.0f))
		.HAlign(Args._bRightSide ? HAlign_Right : HAlign_Left)
		.VAlign(VAlign_Top)
		[
			ListView.ToSharedRef()
		];
		
	ChildSlot.AttachWidget(ListContainer.ToSharedRef());
}

SDebugMenuWidget::~SDebugMenuWidget()
{

}

void SDebugMenuWidget::SetSelectedListItem(TSharedPtr<FDebugMenuListEntryData> Item)
{
	//FSlateApplication::Get().GetUser(0)->CloseTooltip();

	if (!ListView.IsValid())
	{
		return;
	}

	TSharedPtr<FDebugMenuListEntryData> PrevSelected = GetSelectedEntry();
	if (PrevSelected.IsValid())
	{
		UDebugMenuEntry* PrevMenuEntry = PrevSelected->EntryData.Get();
		if (PrevMenuEntry)
		{
			PrevMenuEntry->OnEntryUnhighlighted();
		}
	}

	ListView->SetSelection(Item);
	if (!ListView->IsItemVisible(Item))
	{
		ListView->RequestScrollIntoView(Item);
	}

	UDebugMenuEntry* MenuEntry = Item->EntryData.Get();
	if (MenuEntry)
	{
		MenuEntry->OnEntryHighlighted();
	}

	/*TSharedPtr < ITableRow> TableRow = ListView->WidgetFromItem(Item);
	if (TableRow.IsValid())
	{
		FVector2D RowPos = TableRow->AsWidget()->GetTickSpaceGeometry().GetAbsolutePosition();


		TSharedRef<SToolTip> NewTooltip =
			SNew(SToolTip).Text(LOCTEXT("Blabhbla", "Hehe Tooltip"))
			;
		FSlateApplication::Get().GetUser(0)->ShowTooltip(NewTooltip, RowPos);
	}*/
}

void SDebugMenuWidget::SelectUp()
{
	if (!ListView.IsValid())
	{
		return;
	}

	TArray<TSharedPtr<FDebugMenuListEntryData>> SelectedItems = ListView->GetSelectedItems();
	if (SelectedItems.Num() < 1)
	{
		if (DisplayedEntries.Num() > 0)
		{
			SetSelectedListItem(DisplayedEntries[0]);
		}
		return;
	}

	int32 NumEntries = DisplayedEntries.Num();
	for (int32 i = 0; i < NumEntries; i++)
	{
		if (DisplayedEntries[i] == SelectedItems[0])
		{
			if (i == 0)
			{
				// select the last item in the list
				SetSelectedListItem(DisplayedEntries[NumEntries - 1]);
			}
			else
			{
				// select the previous item
				SetSelectedListItem(DisplayedEntries[i - 1]);
			}
			return;
		}
	}

}

void SDebugMenuWidget::SelectDown()
{
	if (!ListView.IsValid())
	{
		return;
	}

	TArray<TSharedPtr<FDebugMenuListEntryData>> SelectedItems = ListView->GetSelectedItems();
	if (SelectedItems.Num() < 1)
	{
		if (DisplayedEntries.Num() > 0)
		{
			SetSelectedListItem(DisplayedEntries[0]);
		}
		return;
	}

	int32 NumEntries = DisplayedEntries.Num();
	for (int32 i = 0; i < NumEntries; i++)
	{
		if (DisplayedEntries[i] == SelectedItems[0])
		{
			if (i == NumEntries - 1)
			{
				// select the first item in the list
				SetSelectedListItem(DisplayedEntries[0]);
			}
			else
			{
				// select the next item
				SetSelectedListItem(DisplayedEntries[i + 1]);
			}
			return;
		}
	}
}

int32 SDebugMenuWidget::GetSelectedIndex() const
{
	TSharedPtr<FDebugMenuListEntryData> SelectedItem = GetSelectedEntry();

	if (SelectedItem.IsValid() && ListView.IsValid())
	{
		for (int32 i = 0; i < DisplayedEntries.Num(); i++)
		{
			if (DisplayedEntries[i] == SelectedItem)
			{
				return i;
			}
		}
	}

	return -1;
}

TSharedPtr<FDebugMenuListEntryData> SDebugMenuWidget::GetSelectedEntry() const
{
	if (ListView.IsValid())
	{
		TArray<TSharedPtr<FDebugMenuListEntryData>> SelectedItems = ListView->GetSelectedItems();
		if (SelectedItems.Num() > 0)
		{
			const auto& SelectedItem = SelectedItems[0];
			return SelectedItem;
		}
	}

	return nullptr;
}

void SDebugMenuWidget::RequestHideMenu()
{
	TSharedPtr<FDebugMenuListEntryData> PrevSelected = GetSelectedEntry();
	if (PrevSelected.IsValid())
	{
		UDebugMenuEntry* PrevMenuEntry = PrevSelected->EntryData.Get();
		if (PrevMenuEntry)
		{
			PrevMenuEntry->OnEntryUnhighlighted();
		}
	}

	SetVisibility(EVisibility::Collapsed);
}

void SDebugMenuWidget::RefreshEntries(bool bSelectCrumb)
{
	if (!ListView.IsValid())
	{
		return;
	}

	int32 SelectedFocusIndex = 0;
	TSharedPtr<FDebugMenuListEntryData> SelectedItem;
	TArray<TSharedPtr<FDebugMenuListEntryData>> SelectedItems = ListView->GetSelectedItems();
	if (SelectedItems.Num() > 0)
	{
		SelectedItem = SelectedItems[0];
		SelectedFocusIndex = SelectedItem->FocusWidgetIndex;
	}

	DisplayedEntries.Empty();

	ADebugMenuManager* Manager = AssociatedMenuManager.Get();
	if (Manager == nullptr)
	{
		return;
	}

	int32 HierarchyIndex = Manager->GetCurrentHierarchyDisplayIndex();
	if (HierarchyIndex >= 0) // on clients this can be called before beginplay has been triggered
	{
		FDebugMenuHierarchyNode& HierarchyNode = Manager->GetHierarchyNode(HierarchyIndex);
		for (int32 ChildNodeIndex : HierarchyNode.ChildNodes)
		{
			FDebugMenuHierarchyNode& ChildNode = Manager->GetHierarchyNode(ChildNodeIndex);

			TSharedPtr<FDebugMenuListEntryData> NewEntry = MakeShareable(new FDebugMenuListEntryData());
			NewEntry->LibrariesData = ChildNode.Libraries;
			NewEntry->CategoryDisplayName = ChildNode.CategoryName;
			DisplayedEntries.Add(NewEntry);
		}
	}

	// sort categories in order. leave entries in the order defined
	DisplayedEntries.Sort([](const TSharedPtr<FDebugMenuListEntryData>& Lhs, const TSharedPtr<FDebugMenuListEntryData>& Rhs) {
		return Lhs->CategoryDisplayName.LexicalLess(Rhs->CategoryDisplayName);
		});

	Manager->OnCategoryEntriesDisplayed();

	int32 InsertIndex = 0;

	TArray<TWeakObjectPtr<UDebugMenuEntry>> DisplayEntries = Manager->GetCurrentDisplayedEntries();
	for (TWeakObjectPtr<UDebugMenuEntry> EntryWPtr : DisplayEntries)
	{
		UDebugMenuEntry* EntryDef = EntryWPtr.Get();
		if (EntryDef && EntryDef->ShouldShowEntry())
		{
			TSharedPtr<FDebugMenuListEntryData> NewEntry = MakeShareable(new FDebugMenuListEntryData());
			NewEntry->EntryData = EntryWPtr;

			if (EntryDef->bShowAboveCategories)
			{
				// add to the top
				DisplayedEntries.Insert(NewEntry, InsertIndex);
				InsertIndex++;
			}
			else
			{
				DisplayedEntries.Add(NewEntry);
			}
		}
	}

	TSharedPtr<FDebugMenuListEntryData> BackEntry = MakeShareable(new FDebugMenuListEntryData());
	BackEntry->bIsGoBack = true;
	DisplayedEntries.Add(BackEntry);

	if (ListView.IsValid())
	{
		ListView->RebuildList();
		if (DisplayedEntries.Num() > 0)
		{
			bool bFoundPreviousSelected = false;
			if (SelectedItem.IsValid())
			{
				for (TSharedPtr<FDebugMenuListEntryData> CurrEntry : DisplayedEntries)
				{
					if (CurrEntry.IsValid()
						&& CurrEntry->LibrariesData == SelectedItem->LibrariesData
						&& CurrEntry->CategoryDisplayName == SelectedItem->CategoryDisplayName
						&& (CurrEntry->EntryData == SelectedItem->EntryData 
							|| (CurrEntry->EntryData.Get() && CurrEntry->EntryData->IsEqualEntry(SelectedItem->EntryData.Get())))
						&& !CurrEntry->bIsGoBack)
					{
						bFoundPreviousSelected = true;
						SetSelectedListItem(CurrEntry);
						CurrEntry->FocusWidgetIndex = SelectedFocusIndex;
					}
				}
			}

			if (!bFoundPreviousSelected)
			{
				SetSelectedListItem(DisplayedEntries[0]);
			}

			if (bSelectCrumb)
			{
				int32 Crumb = Manager->GetSelectionCrumb();
				if (Crumb >= 0 && DisplayedEntries.Num() > Crumb)
				{
					SetSelectedListItem(DisplayedEntries[Crumb]);
				}
			}
		}
	}
}

TSharedRef<ITableRow> SDebugMenuWidget::OnGenerateRow(TSharedPtr<FDebugMenuListEntryData> Item, const TSharedRef<STableViewBase>& Owner)
{
	if (!Item.IsValid())
	{
		return SNew(STableRow< TSharedPtr<FDebugMenuListEntryData>> , Owner)
			[
				SNew(STextBlock)
				.Text(FText::FromString(FString("THIS WAS NULL SOMEHOW")))
			];
	}

	FSlateBrush TransparentBrush;
	TransparentBrush.TintColor = FSlateColor(MenuBGColor);

	TransparentTableRowStyle = FTableRowStyle::GetDefault();
	TransparentTableRowStyle.SetEvenRowBackgroundBrush(TransparentBrush);
	TransparentTableRowStyle.SetOddRowBackgroundBrush(TransparentBrush);

	FSlateBrush HighlightedBrush;
	HighlightedBrush.TintColor = FSlateColor(MenuHighlightedBGColor);
	TransparentTableRowStyle.SetInactiveBrush(HighlightedBrush);

	TSharedRef<STableRow< TSharedPtr<FDebugMenuListEntryData> >> result =
		SNew(STableRow< TSharedPtr<FDebugMenuListEntryData> >, Owner)
		.Style(&TransparentTableRowStyle)
		.Padding(MenuRowMargin)
		[
			SNew(SDebugMenuEntry)
			.EntryData(Item)
			.MenuWorld(AssociatedWorld)
			.MenuWidth(MenuWidth)
			.EntryFont(EntryFont)
			.CategoryFont(CategoryFont)
		];

	return result;
}

void SDebugMenuEntry::Construct(const FArguments& args)
{
	AssociatedWorld = args._MenuWorld;

	const float EntryWidth = args._MenuWidth;
	FLinearColor shadowColor = FLinearColor(0, 0, 0, 1.0f);
	FIntPoint shadowOffset = FIntPoint(-2, 2);

	ADebugMenuLibrary* LibraryData = args._EntryData->LibrariesData.Num() > 0 ? args._EntryData->LibrariesData[0].Get() : nullptr;
	if (LibraryData)
	{
		FString EntryName = LibraryData->GetCategoryDisplayName();
		FLinearColor TextColor = FLinearColor(1.0f, 1.0f, 0.4f, 1.0f);
		FLinearColor* color = &TextColor;

		TSharedRef<STextBlock> text = SNew(STextBlock)
			.Text(FText::FromString(EntryName))
			.Font(args._CategoryFont)
			.ColorAndOpacity(*color)
			.ShadowColorAndOpacity(shadowColor)
			.ShadowOffset(shadowOffset);

		TSharedRef<SHorizontalBox> hbox = SNew(SHorizontalBox)
			+ SHorizontalBox::Slot()
			.HAlign(HAlign_Left)
			[
				// name text
				SNew(SBox)
				.WidthOverride(EntryWidth)
				[
					text
				]
			];

		ChildSlot.AttachWidget(hbox);
	}
	else if (UDebugMenuEntry* EntryData = args._EntryData->EntryData.Get())
	{
		ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(EntryData);
		TSharedPtr<SWidget> ConstructedWidget = EntryData->ConstructSlateWidget(EntryWidth, args._EntryData, MenuManager, args._EntryFont);
		TSharedRef<SBorder> WrapBorder = SNew(SBorder)
			.BorderImage_UObject(EntryData, &UDebugMenuEntry::GetBorderImage)
			[
				ConstructedWidget.ToSharedRef()
			];

		ChildSlot.AttachWidget(WrapBorder);
	}
	else if (args._EntryData->bIsGoBack)
	{
		FString EntryName = "[-- go back --]";
		ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(AssociatedWorld.Get());
		if (MenuManager && MenuManager->GetCurrentHierarchyDisplayIndex() == 0)
		{
			EntryName = "[-- close --]";
		}
		
		FLinearColor TextColor = FLinearColor::White;
		FLinearColor* color = &TextColor;

		TSharedRef<STextBlock> text = SNew(STextBlock)
			.Text(FText::FromString(EntryName))
			.Font(args._EntryFont)
			.ColorAndOpacity(*color)
			.ShadowColorAndOpacity(shadowColor)
			.ShadowOffset(shadowOffset);

		TSharedRef<SHorizontalBox> hbox = SNew(SHorizontalBox)
			+ SHorizontalBox::Slot()
			.HAlign(HAlign_Left)
			[
				// name text
				SNew(SBox)
				.WidthOverride(EntryWidth)
			[
				text
			]
			];

		ChildSlot.AttachWidget(hbox);
	}
	
	
}

SDebugMenuEntry::~SDebugMenuEntry()
{

}

FReply SDebugMenuEntry::OnMouseButtonDown(const FGeometry& MyGeometry, const FPointerEvent& MouseEvent)
{
	// support clicking on entries
	UWorld* World = AssociatedWorld.Get();
	if (World)
	{
		World->GetTimerManager().SetTimerForNextTick([this, World]()
		{
			ADebugMenuManager* MenuManager = ADebugMenuManager::GetDebugMenuManager(World);
			if (MenuManager)
			{
				MenuManager->InputConfirm();
			}
		});
	}

	return SCompoundWidget::OnMouseButtonDown(MyGeometry, MouseEvent);
}

#undef LOCTEXT_NAMESPACE
