// Copyright (c) 2022 FWORKS. All Rights Reserved.

#pragma once

#include "CoreMinimal.h"
#include "GameFramework/Actor.h"
#include "Engine/DataAsset.h"
#include "Slate.h"
#include "DebugMenuTypes.h"
#include "DebugMenuEntry.generated.h"

class FDebugMenuListEntryData;
class ADebugMenuManager;

UCLASS(BlueprintType, Abstract, EditInlineNew, DefaultToInstanced, CollapseCategories)
class DEBUGMENU_API UDebugMenuEntry : public UObject
{
	GENERATED_BODY()
	
public:	
	UDebugMenuEntry();
	virtual FString GetDisplayString() const { return ""; }
	virtual void OnInputToggleWatch(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) { }
	virtual bool OnInputConfirm(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) { return false; }
	virtual bool OnInputLeft(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) { return false; }
	virtual bool OnInputRight(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) { return false; }
	virtual bool OnInputCancel(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) { return false; }
	virtual FLinearColor GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const { return FLinearColor::White; }
	virtual TSharedPtr<SWidget> ConstructSlateWidget(float EntryWidth, TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager, const FSlateFontInfo& EntryFont);
	virtual bool IsEqualEntry(UDebugMenuEntry* OtherEntry) const;
	virtual const FSlateBrush* GetBorderImage() const;

	virtual void OnEntryHighlighted() {}
	virtual void OnEntryUnhighlighted() {}
	virtual void OnEntryDisplayBegin() {}
	virtual void OnEntryDisplayEnd() {}
	bool bShowAboveCategories = false;

	virtual bool ShouldShowEntry() const { return true; }

	void SetDisplayStringPrefix(FString NewPrefix) { DisplayStringPrefix = NewPrefix; }
	FString GetDisplayStringPrefix() const { return DisplayStringPrefix; }
	FString DisplayStringPrefix = "";
};

// plain text label
UCLASS(BlueprintType, DisplayName = "Text Label", EditInlineNew, CollapseCategories)
class DEBUGMENU_API UDebugMenuEntryTextLabel : public UDebugMenuEntry
{
	GENERATED_BODY()

public:
	UDebugMenuEntryTextLabel() {}
	void SetTextLabel(FString NewString) { TextLabel = NewString; }
	virtual const FSlateBrush* GetBorderImage() const override;
	virtual void OnInputToggleWatch(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;

	TWeakObjectPtr<UObject> TargetObject;
	TFieldPath<FProperty> TargetProperty;
	TArray<FDebugMenuPropertyChain> PropertyChain;

protected:

	virtual FString GetDisplayString() const override { return TextLabel; }

	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FString TextLabel = "--------";
};

// debugmenu entry defined by functions in the ADebugMenuLibrary blueprint
UCLASS(BlueprintType, DisplayName = "Blueprint Function", EditInlineNew, CollapseCategories)
class DEBUGMENU_API UDebugMenuEntryBlueprint : public UDebugMenuEntry
{
	GENERATED_BODY()

public:
	UDebugMenuEntryBlueprint();

protected:

	virtual FString GetDisplayString() const override;
	virtual bool OnInputConfirm(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual bool OnInputLeft(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual bool OnInputRight(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual FLinearColor GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const override;

public:

	// use this as the menu display string, unless FunctionName_GetDisplayString is available
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FString DefaultDisplayString = "";

	// Function must have no inputs, and have a single output of type "String" named "ReturnValue"
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_GetDisplayString;

	// use this as the menu display color, unless FunctionName_GetColor is available
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FLinearColor DefaultDisplayColor = FLinearColor::White;

	// Function must have no inputs, and have a single output of type "LinearColor" named "ReturnValue"
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_GetColor;

	// Function called when CONFIRM input is pressed. It must have no input and output parameters
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_OnExecute;

	// Function called when LEFT input is pressed. It must have no input and output parameters
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_OnInputLeft;

	// Function called when RIGHT input is pressed. It must have no input and output parameters
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_OnInputRight;
};
