// Copyright (c) 2022 FWORKS. All Rights Reserved.

#pragma once

#include "CoreMinimal.h"
#include "GameFramework/Actor.h"
#include "Engine/DataAsset.h"
#include "Slate.h"
#include "DebugMenuEntry.h"
#include "DebugMenuTypes.h"
#include "UObject/UObjectGlobals.h"
#include "DebugMenuEntryBoolBase.generated.h"

class FDebugMenuListEntryData;
class ADebugMenuManager;

DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnDebugMenuBoolValueChanged, bool, NewValue);

UCLASS(Abstract, BlueprintType, DisplayName = "Bool Entry Base", CollapseCategories, EditInlineNew)
class DEBUGMENU_API UDebugMenuEntryBoolBase : public UDebugMenuEntry
{
	GENERATED_BODY()

public:
	UDebugMenuEntryBoolBase();

	// Color when the value is FALSE
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FLinearColor ColorWhenFalse = FLinearColor::White;

	// Color when the value is TRUE
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FLinearColor ColorWhenTrue = FLinearColor::Green;

	virtual FString GetDisplayString() const override { return ""; }
	virtual bool OnInputLeft(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual bool OnInputRight(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual bool OnInputConfirm(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual bool OnInputCancel(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual FLinearColor GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const override;
	
	virtual void SetBoolValue(bool NewValue) {}
	virtual bool GetBoolValue() const { return false; }

	UPROPERTY(BlueprintAssignable, Transient, Category = "DebugMenu Entry")
	FOnDebugMenuBoolValueChanged OnValueChanged;
};

// Menu entry to manipulate a boolean cvar
UCLASS(BlueprintType, DisplayName="Bool CVar", CollapseCategories, EditInlineNew)
class DEBUGMENU_API UDebugMenuEntryBoolCVar : public UDebugMenuEntryBoolBase
{
	GENERATED_BODY()

public:
	UDebugMenuEntryBoolCVar();

	virtual FString GetDisplayString() const override;
	virtual FLinearColor GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const override;
	virtual bool GetBoolValue() const override;
	virtual void SetBoolValue(bool NewValue) override;
	virtual bool ShouldShowEntry() const override;
protected:

	// The CVar that will be controlled
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FString CVarString = "";

	// String that will be shown for this menu entry
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FString DisplayString = "";

	// Text Color when CVar does not exist
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FLinearColor ColorWhenInvalid = FLinearColor::Gray;
};

UCLASS(BlueprintType, DisplayName = "Bool UObject Prop", CollapseCategories, EditInlineNew, HideDropdown)
class DEBUGMENU_API UDebugMenuEntryBoolUObjectProp : public UDebugMenuEntryBoolBase
{
	GENERATED_BODY()

public:
	UDebugMenuEntryBoolUObjectProp();
	virtual FString GetDisplayString() const override;
	virtual bool GetBoolValue() const override;
	virtual void SetBoolValue(bool NewValue) override;
	virtual bool IsEqualEntry(UDebugMenuEntry* OtherEntry) const override;
	virtual const FSlateBrush* GetBorderImage() const override;
	virtual void OnInputToggleWatch(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	TWeakObjectPtr<UObject> TargetObject;
	TFieldPath<FProperty> TargetProperty;
	TArray<FDebugMenuPropertyChain> PropertyChain;
};

// Menu entry to manipulate a stat command, eg "stat fps" command
UCLASS(BlueprintType, DisplayName = "Bool EngineStat", CollapseCategories, EditInlineNew)
class DEBUGMENU_API UDebugMenuEntryBoolEngineStat : public UDebugMenuEntryBoolBase
{
	GENERATED_BODY()

public:
	UDebugMenuEntryBoolEngineStat();

	virtual FString GetDisplayString() const override;
	virtual bool GetBoolValue() const override;
	virtual void SetBoolValue(bool NewValue) override;

protected:

	// The stat that will be controlled (do not include the stat prefix)
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FString StatString = "";

	// String that will be shown for this menu entry
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FString DisplayString = "";
};

// menu entry to manipulate a showflag command, eg "showflag.staticmeshes"
UCLASS(BlueprintType, DisplayName = "Bool ShowFlag", CollapseCategories, EditInlineNew)
class DEBUGMENU_API UDebugMenuEntryBoolShowFlag : public UDebugMenuEntryBoolBase
{
	GENERATED_BODY()

public:
	UDebugMenuEntryBoolShowFlag();

	virtual FString GetDisplayString() const override;
	virtual bool GetBoolValue() const override;
	virtual void SetBoolValue(bool NewValue) override;

protected:

	// The showflag that will be controlled (do not include the showflag prefix)
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FString ShowFlagString = "";

	// String that will be shown for this menu entry
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FString DisplayString = "";
};

// menu entry to manipulate a user defined boolean whose getter and setter are defined in the library blueprint
UCLASS(BlueprintType, DisplayName = "Bool Blueprint", CollapseCategories, EditInlineNew)
class DEBUGMENU_API UDebugMenuEntryBoolBlueprint : public UDebugMenuEntryBoolBase
{
	GENERATED_BODY()

public:
	UDebugMenuEntryBoolBlueprint() {}

	virtual FString GetDisplayString() const override;
	virtual bool GetBoolValue() const override;
	virtual void SetBoolValue(bool NewValue) override;

protected:

	// use this as the menu display text, unless FunctionName_GetDisplayString is available
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FString DefaultDisplayString = "";

	// Function must have no inputs, and have a single output of type "String" named "ReturnValue"
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_GetDisplayString;

	// Function must have no inputs, and have a single output of type "boolean" named "ReturnValue"
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_GetBoolValue;

	// Function must have no outputs, and have a single input of type "boolean" named "Input"
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_SetBoolValue;
};
