// Copyright (c) 2022 FWORKS. All Rights Reserved.

#pragma once

#include "CoreMinimal.h"
#include "GameFramework/Actor.h"
#include "Engine/DataAsset.h"
#include "Slate.h"
#include "DebugMenuEntry.h"
#include "DebugMenuTypes.h"
#include "DebugMenuEntryFloatBase.generated.h"

class FDebugMenuListEntryData;
class ADebugMenuManager;

UCLASS(Abstract, BlueprintType, DisplayName = "Float Entry Base", CollapseCategories, EditInlineNew)
class DEBUGMENU_API UDebugMenuEntryFloatBase : public UDebugMenuEntry
{
	GENERATED_BODY()

public:
	UDebugMenuEntryFloatBase();

	virtual FString GetDisplayString() const override { return ""; }
	virtual bool OnInputLeft(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual bool OnInputRight(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual bool OnInputConfirm(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual bool OnInputCancel(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual FLinearColor GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const override { return FLinearColor::White; }
	virtual TSharedPtr<SWidget> ConstructSlateWidget(float EntryWidth, TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager, const FSlateFontInfo& EntryFont) override;

	virtual void SetFloatValue(float NewValue) {}
	virtual float GetFloatValue() const { return 0.0f; }

	virtual TOptional<float> GetFloatBoxValue() const;
	virtual void OnValueCommited(float InValue, ETextCommit::Type InCommitType);
	virtual void OnUndeterminedValueCommited(FText NewValue, ETextCommit::Type CommitType);

protected:
	// Limit the range of the float
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	bool bLimitRange = false;

	// Minimum range of the float
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry", meta = (EditCondition = "bLimitRange", EditConditionHides))
	float RangeMin = 0.0f;

	// Maximum range of the float
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry", meta = (EditCondition = "bLimitRange", EditConditionHides))
	float RangeMax = 1.0f;

	// How much the float changes when you press the left/right input keys
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	float TweakInterval = 0.1f;
};

// menu entry to manipulate a float cvar
UCLASS(BlueprintType, DisplayName="Float CVar", CollapseCategories, EditInlineNew)
class DEBUGMENU_API UDebugMenuEntryFloatCVar : public UDebugMenuEntryFloatBase
{
	GENERATED_BODY()

public:
	UDebugMenuEntryFloatCVar();

	virtual FString GetDisplayString() const override;
	virtual FLinearColor GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const override;
	virtual float GetFloatValue() const override;
	virtual void SetFloatValue(float NewValue) override;
	virtual bool ShouldShowEntry() const override;

protected:

	// The CVar that will be controlled
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FString CVarString = "";

	// String that will be shown for this menu entry
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FString DisplayString = "";

	// Text Color when CVar is at minimum range
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FLinearColor TextColorRangeMin = FLinearColor::White;

	// Text Color when CVar is at maximum range
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FLinearColor TextColorRangeMax = FLinearColor::White;

	// Text Color when CVar does not exist
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FLinearColor ColorWhenInvalid = FLinearColor::Gray;
};

// menu entry to manipulate a generic float whose getter and setter are defined in the library blueprint
UCLASS(BlueprintType, DisplayName = "Float Blueprint", CollapseCategories, EditInlineNew)
class DEBUGMENU_API UDebugMenuEntryFloatBlueprint : public UDebugMenuEntryFloatBase
{
	GENERATED_BODY()

public:
	UDebugMenuEntryFloatBlueprint() {}

	virtual FString GetDisplayString() const override;
	virtual FLinearColor GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const override;
	virtual float GetFloatValue() const override;
	virtual void SetFloatValue(float NewValue) override;

protected:

	// use this as the menu display text, unless FunctionName_GetDisplayString is available
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FString DefaultDisplayString = "";

	// Function must have no inputs, and have a single output of type "String" named "ReturnValue"
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_GetDisplayString;

	// use this as the menu display color, unless FunctionName_GetColor is available
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FLinearColor DefaultDisplayColor = FLinearColor::White;

	// Function must have no inputs, and have a single output of type "LinearColor" named "ReturnValue"
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_GetColor;

	// Function must have no inputs, and have a single output of type "float" named "ReturnValue"
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_GetFloatValue;

	// Function must have no outputs, and have a single input of type "float" named "Input"
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_SetFloatValue;
};

UCLASS(BlueprintType, DisplayName="Float UObject Prop", CollapseCategories, EditInlineNew, HideDropdown)
class DEBUGMENU_API UDebugMenuEntryFloatUObjectProp : public UDebugMenuEntryFloatBase
{
	GENERATED_BODY()

public:
	UDebugMenuEntryFloatUObjectProp();
	virtual FString GetDisplayString() const override;
	virtual FLinearColor GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const override;
	virtual float GetFloatValue() const override;
	virtual void SetFloatValue(float NewValue) override;
	virtual bool IsEqualEntry(UDebugMenuEntry* OtherEntry) const override;
	virtual const FSlateBrush* GetBorderImage() const override;
	virtual void OnInputToggleWatch(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	TWeakObjectPtr<UObject> TargetObject;
	TFieldPath<FProperty> TargetProperty;
	TArray<FDebugMenuPropertyChain> PropertyChain;
};