// Copyright (c) 2022 FWORKS. All Rights Reserved.

#pragma once

#include "CoreMinimal.h"
#include "GameFramework/Actor.h"
#include "Engine/DataAsset.h"
#include "Slate.h"
#include "DebugMenuEntry.h"
#include "DebugMenuTypes.h"
#include "DebugMenuEntryIntBase.generated.h"

class FDebugMenuListEntryData;
class ADebugMenuManager;

DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnDebugMenuIntValueChanged, int32, NewValue);

UCLASS(Abstract, BlueprintType, DisplayName = "Int Entry Base", CollapseCategories, EditInlineNew)
class DEBUGMENU_API UDebugMenuEntryIntBase : public UDebugMenuEntry
{
	GENERATED_BODY()

public:
	UDebugMenuEntryIntBase();

	virtual FString GetDisplayString() const override { return ""; }
	virtual bool OnInputLeft(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual bool OnInputRight(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual bool OnInputConfirm(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual bool OnInputCancel(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual FLinearColor GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const override { return FLinearColor::White; }
	virtual TSharedPtr<SWidget> ConstructSlateWidget(float EntryWidth, TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager, const FSlateFontInfo& EntryFont) override;

	virtual void SetIntValue(int32 NewValue) {}
	virtual int32 GetIntValue() const { return 0; }

	virtual TOptional<int32> GetIntBoxValue() const;
	virtual void OnValueCommited(int32 InValue, ETextCommit::Type InCommitType);
	virtual void OnUndeterminedValueCommited(FText NewValue, ETextCommit::Type CommitType);

	void SetTweakInterval(int32 NewInterval) { TweakInterval = NewInterval; }

	UPROPERTY(BlueprintAssignable, Transient, Category = "DebugMenu Entry")
	FOnDebugMenuIntValueChanged OnValueChanged;

protected:
	// Limit the range of the integer
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	bool bLimitRange = false;

	// Minimum range of the integer
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry", meta = (EditCondition = "bLimitRange", EditConditionHides))
	int32 RangeMin = 0;

	// Maximum range of the integer
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry", meta = (EditCondition = "bLimitRange", EditConditionHides))
	int32 RangeMax = 1;

	// How much the integer changes when you press the left/right input keys
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	int32 TweakInterval = 1;

	// whether to display this integer as a bool (limited 0 to 1)
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	bool bUseBoolRepresentation = false;
};

// menu entry to manipulate an integer cvar
UCLASS(BlueprintType, DisplayName="Int32 CVar", CollapseCategories, EditInlineNew)
class DEBUGMENU_API UDebugMenuEntryIntCVar : public UDebugMenuEntryIntBase
{
	GENERATED_BODY()

public:
	UDebugMenuEntryIntCVar();

	virtual FString GetDisplayString() const override;
	virtual FLinearColor GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const override;
	virtual int32 GetIntValue() const override;
	virtual void SetIntValue(int32 NewValue) override;
	virtual bool ShouldShowEntry() const override;
protected:

	// The CVar that will be controlled
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FString CVarString = "";

	// String that will be shown for this menu entry
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FString DisplayString = "";

	// Text Color when CVar is at minimum range
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FLinearColor TextColorRangeMin = FLinearColor::White;

	// Text Color when CVar is at maximum range
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FLinearColor TextColorRangeMax = FLinearColor::White;

	// Text Color when CVar does not exist
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FLinearColor ColorWhenInvalid = FLinearColor::Gray;
};

// menu entry to manipulate a generic integer, whose getter and setter are defined in the libary blueprint
UCLASS(BlueprintType, DisplayName = "Int32 Blueprint", CollapseCategories, EditInlineNew)
class DEBUGMENU_API UDebugMenuEntryIntBlueprint : public UDebugMenuEntryIntBase
{
	GENERATED_BODY()

public:
	UDebugMenuEntryIntBlueprint() {}

	virtual FString GetDisplayString() const override;
	virtual FLinearColor GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const override;
	virtual int32 GetIntValue() const override;
	virtual void SetIntValue(int32 NewValue) override;

protected:

	// use this as the menu display text, unless FunctionName_GetDisplayString is available
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FString DefaultDisplayString = "";

	// Function must have no inputs, and have a single output of type "String" named "ReturnValue"
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_GetDisplayString;

	// use this as the menu display color, unless FunctionName_GetColor is available
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FLinearColor DefaultDisplayColor = FLinearColor::White;

	// Function must have no inputs, and have a single output of type "LinearColor" named "ReturnValue"
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_GetColor;

	// Function must have no inputs, and have a single output of type "Integer" named "ReturnValue"
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_GetIntValue;

	// Function must have no outputs, and have a single input of type "Integer" named "Input"
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_SetIntValue;
};

UCLASS(BlueprintType, DisplayName = "Int32 UObject Prop", CollapseCategories, EditInlineNew, HideDropdown)
class DEBUGMENU_API UDebugMenuEntryIntUObjectProp : public UDebugMenuEntryIntBase
{
	GENERATED_BODY()

public:
	UDebugMenuEntryIntUObjectProp();
	virtual FString GetDisplayString() const override;
	virtual FLinearColor GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const override;
	virtual int32 GetIntValue() const override;
	virtual void SetIntValue(int32 NewValue) override;
	virtual bool IsEqualEntry(UDebugMenuEntry* OtherEntry) const override;
	virtual const FSlateBrush* GetBorderImage() const override;
	virtual void OnInputToggleWatch(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	TWeakObjectPtr<UObject> TargetObject;
	TFieldPath<FProperty> TargetProperty;
	TArray<FDebugMenuPropertyChain> PropertyChain;
};