// Copyright (c) 2022 FWORKS. All Rights Reserved.

#pragma once

#include "CoreMinimal.h"
#include "GameFramework/Actor.h"
#include "Engine/DataAsset.h"
#include "Slate.h"
#include "DebugMenuEntry.h"
#include "DebugMenuTypes.h"
#include "DebugMenuEntryRotatorBase.generated.h"

class FDebugMenuListEntryData;
class ADebugMenuManager;

// menu entry for manipulating Rotators
UCLASS(Abstract, BlueprintType, DisplayName = "Rotator Entry Base", CollapseCategories, EditInlineNew)
class DEBUGMENU_API UDebugMenuEntryRotatorBase : public UDebugMenuEntry
{
	GENERATED_BODY()

public:
	UDebugMenuEntryRotatorBase();

	virtual FString GetDisplayString() const override { return ""; }
	virtual bool OnInputLeft(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual bool OnInputRight(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual bool OnInputConfirm(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual bool OnInputCancel(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual FLinearColor GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const override { return FLinearColor::White; }
	virtual TSharedPtr<SWidget> ConstructSlateWidget(float EntryWidth, TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager, const FSlateFontInfo& EntryFont) override;

	virtual void SetRotatorValue(FRotator NewValue) {}
	virtual FRotator GetRotatorValue() const { return FRotator::ZeroRotator; }

	TOptional<float> GetValueX() const { return GetRotatorValue().Pitch; }
	TOptional<float> GetValueY() const { return GetRotatorValue().Yaw; }
	TOptional<float> GetValueZ() const { return GetRotatorValue().Roll; }

	virtual void OnValueCommitedX(float InValue, ETextCommit::Type InCommitType);
	virtual void OnValueCommitedY(float InValue, ETextCommit::Type InCommitType);
	virtual void OnValueCommitedZ(float InValue, ETextCommit::Type InCommitType);
	void OnUndeterminedValueCommited(FText NewValue, ETextCommit::Type CommitType);

protected:
	
};

// menu entry to manipulate a generic Rotator whose getter and setter are defined in the library blueprint
UCLASS(BlueprintType, DisplayName = "Rotator Blueprint", CollapseCategories, EditInlineNew)
class DEBUGMENU_API UDebugMenuEntryRotatorBlueprint : public UDebugMenuEntryRotatorBase
{
	GENERATED_BODY()

public:
	UDebugMenuEntryRotatorBlueprint() {}

	virtual FString GetDisplayString() const override;
	virtual FLinearColor GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const override;
	virtual FRotator GetRotatorValue() const override;
	virtual void SetRotatorValue(FRotator NewValue) override;

protected:

	// use this as the menu display text, unless FunctionName_GetDisplayString is available
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FString DefaultDisplayString = "";

	// Function must have no inputs, and have a single output of type "String" named "ReturnValue"
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_GetDisplayString;

	// use this as the menu display color, unless FunctionName_GetColor is available
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FLinearColor DefaultDisplayColor = FLinearColor::White;

	// Function must have no inputs, and have a single output of type "LinearColor" named "ReturnValue"
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_GetColor;

	// Function must have no inputs, and have a single output of type "Rotator" named "ReturnValue"
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_GetRotatorValue;

	// Function must have no outputs, and have a single input of type "Rotator" named "Input"
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FName FunctionName_SetRotatorValue;
};

UCLASS(BlueprintType, DisplayName = "Rotator UObject Prop", CollapseCategories, EditInlineNew, HideDropdown)
class DEBUGMENU_API UDebugMenuEntryRotatorUObjectProp : public UDebugMenuEntryRotatorBase
{
	GENERATED_BODY()

public:
	UDebugMenuEntryRotatorUObjectProp();
	virtual FString GetDisplayString() const override;
	virtual FLinearColor GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const override;
	virtual FRotator GetRotatorValue() const override;
	virtual void SetRotatorValue(FRotator NewValue) override;
	virtual bool IsEqualEntry(UDebugMenuEntry* OtherEntry) const override;
	virtual const FSlateBrush* GetBorderImage() const override;
	virtual void OnInputToggleWatch(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	TWeakObjectPtr<UObject> TargetObject;
	TFieldPath<FProperty> TargetProperty;
	TArray<FDebugMenuPropertyChain> PropertyChain;
};