// Copyright (c) 2022 FWORKS. All Rights Reserved.

#pragma once

#include "CoreMinimal.h"
#include "GameFramework/Actor.h"
#include "Engine/DataAsset.h"
#include "Slate.h"
#include "DebugMenuEntry.h"
#include "DebugMenuTypes.h"
#include "DebugMenuEntryStringBase.generated.h"

class FDebugMenuListEntryData;
class ADebugMenuManager;

DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnDebugMenuStringEntryCommitted, FString, NewValue);

// menu entry for manipulating string data
UCLASS(Abstract, BlueprintType, DisplayName = "String Entry Base", CollapseCategories, EditInlineNew)
class DEBUGMENU_API UDebugMenuEntryStringBase : public UDebugMenuEntry
{
	GENERATED_BODY()

public:
	UDebugMenuEntryStringBase();

	virtual FString GetDisplayString() const override { return ""; }
	virtual bool OnInputLeft(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual bool OnInputRight(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual bool OnInputConfirm(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual bool OnInputCancel(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	virtual FLinearColor GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const override { return DisplayColor; }
	virtual TSharedPtr<SWidget> ConstructSlateWidget(float EntryWidth, TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager, const FSlateFontInfo& EntryFont) override;

	virtual void SetStringValue(FString NewValue) {}
	virtual FString GetStringValue() const { return ""; }

	void OnValueCommited(const FText& InNewText, ETextCommit::Type InCommitType);
	virtual FText GetStringBoxValue() const;

	// display color for the description
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FLinearColor DisplayColor = FLinearColor::White;

	// delegate when user commits the textbox
	UPROPERTY(BlueprintAssignable, Transient, Category = "DebugMenu Entry")
	FOnDebugMenuStringEntryCommitted OnStringCommitted;

protected:
	
};

// menu entry to manipulate an FString cvar
UCLASS(BlueprintType, DisplayName="String CVar", CollapseCategories, EditInlineNew)
class DEBUGMENU_API UDebugMenuEntryStringCVar : public UDebugMenuEntryStringBase
{
	GENERATED_BODY()

public:
	UDebugMenuEntryStringCVar();

	virtual FString GetDisplayString() const override;
	virtual FLinearColor GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const override;
	virtual FString GetStringValue() const override;
	virtual void SetStringValue(FString NewValue) override;

protected:

	// The CVar that will be controlled
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FString CVarString = "";

	// String that will be shown for this menu entry
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FString DisplayString = "";

	// Text Color when CVar does not exist
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entry")
	FLinearColor ColorWhenInvalid = FLinearColor::Gray;
};

// menu entry to manipulate a string uproperty in a uobject
UCLASS(BlueprintType, DisplayName = "String UObject Prop", CollapseCategories, EditInlineNew, HideDropdown)
class DEBUGMENU_API UDebugMenuEntryStringUObjectProp : public UDebugMenuEntryStringBase
{
	GENERATED_BODY()

public:
	UDebugMenuEntryStringUObjectProp();
	virtual FString GetDisplayString() const override;
	virtual FLinearColor GetDisplayColor(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) const override;
	virtual FString GetStringValue() const override;
	virtual void SetStringValue(FString NewValue) override;
	virtual bool IsEqualEntry(UDebugMenuEntry* OtherEntry) const override;
	virtual const FSlateBrush* GetBorderImage() const override;
	virtual void OnInputToggleWatch(TSharedPtr<FDebugMenuListEntryData> RuntimeEntryData, ADebugMenuManager* MenuManager) override;
	TWeakObjectPtr<UObject> TargetObject;
	TFieldPath<FProperty> TargetProperty;
	TArray<FDebugMenuPropertyChain> PropertyChain;
};