// Copyright (c) 2022 FWORKS. All Rights Reserved.

#pragma once

#include "CoreMinimal.h"
#include "Slate.h"
#include "DebugMenuWidget.h"
#include "GameFramework/Actor.h"
#include "DebugMenuTypes.h"
#include "DebugMenuManager.generated.h"

class UDebugMenuEntry;

DECLARE_DYNAMIC_MULTICAST_DELEGATE(FDebugMenuActivationSignature);

struct FDebugMenuObjPropWatch
{
	TWeakObjectPtr<UObject> TargetObject;
	TFieldPath<FProperty> TargetProperty;
	FString DisplayPrefix = "";
	TArray<FDebugMenuPropertyChain> PropertyChain;
};

struct FDebugMenuHierarchyNode
{
	// index of parent in the tree
	int32 ParentNode = -1;

	// category display name
	FName CategoryName;

	// indexes of children in the tree
	TArray<int32> ChildNodes;

	// libraries visible at this level
	TArray< TWeakObjectPtr<ADebugMenuLibrary> > Libraries;
};

UCLASS(BlueprintType, Blueprintable)
class DEBUGMENU_API ADebugMenuManager : public AActor
{
	GENERATED_BODY()
	
public:	
	ADebugMenuManager();
	virtual void BeginPlay() override;
	virtual void EndPlay(const EEndPlayReason::Type EndPlayReason) override;
	virtual void Tick(float DeltaSeconds) override;
	void ClearObjectPropertyWatches();
	void ToggleObjectPropertyWatch(FDebugMenuObjPropWatch WatchData);
	FString GetObjectPropertyValueAsString(const UObject* InObject, FProperty* Property, FString PropertyNameOverride, TArray<FDebugMenuPropertyChain> PropertyChain);
	void RefreshMenu();
	int32 GetCurrentHierarchyDisplayIndex() const;
	void SetCurrentHierarchyDisplayIndex(int32 NewIndex);
	void RegisterLibrary(ADebugMenuLibrary* Library);
	void UnregisterLibrary(ADebugMenuLibrary* Library);
	TArray< TWeakObjectPtr<UDebugMenuEntry> > GetCurrentDisplayedEntries() const;
	bool IsWidgetInChildHierarchy(TSharedPtr<SWidget> Child, TSharedPtr<SWidget> Parent) const;
	FDebugMenuHierarchyNode& GetHierarchyNode(int32 NodeIndex) { return DebugMenuHierarchy[NodeIndex]; }
	void OnCategoryEntriesDisplayed();
	int32 FindHierarchyIndex(TArray<FName> InCategoryHierarchy) const;
	int32 GetSelectionCrumb() const;
	void PushSelectionCrumb();
	void PopSelectionCrumb();
	static void DebugMenuSink();
	void RecreateMenu();
	const FSlateColorBrush& GetWatchedBackgroundBrush() const { return WatchedBackgroundBrush; }
	bool IsPropertyBeingWatched(FDebugMenuObjPropWatch& CheckProperty);

	static void* GetContainerPtrFromPropertyChain(const UObject* InObject, TArray<FDebugMenuPropertyChain> PropertyChain);

	// Notify the menu that DOWN input was pressed
	UFUNCTION(BlueprintCallable, Category = "DebugMenu Plugin")
	void InputDown();

	// Notify the menu that UP input was pressed
	UFUNCTION(BlueprintCallable, Category = "DebugMenu Plugin")
	void InputUp();

	// Notify the menu that ADD WATCH input was pressed
	UFUNCTION(BlueprintCallable, Category = "DebugMenu Plugin")
	void InputToggleWatch();

	// Notify the menu that CONFIRM input was pressed
	UFUNCTION(BlueprintCallable, Category = "DebugMenu Plugin")
	void InputConfirm();

	// Notify the menu that CANCEL input was pressed
	UFUNCTION(BlueprintCallable, Category = "DebugMenu Plugin")
	void InputCancel();

	// Notify the menu that LEFT input was pressed
	UFUNCTION(BlueprintCallable, Category = "DebugMenu Plugin")
	void InputLeft();

	// Notify the menu that RIGHT input was pressed
	UFUNCTION(BlueprintCallable, Category = "DebugMenu Plugin")
	void InputRight();

	// Get the debug menu manager, if it exists in the map
	UFUNCTION(BlueprintCallable, Category = "DebugMenu Plugin")
	static ADebugMenuManager* GetDebugMenuManager(const UObject* WorldContextObj);

	// Toggles the debug menu on/off
	UFUNCTION(BlueprintCallable, Category = "DebugMenu Plugin")
	void ToggleDebugMenu();

	// Shows the debug menu
	UFUNCTION(BlueprintCallable, Category = "DebugMenu Plugin")
	void ShowDebugMenu();

	// Hides the debug menu
	UFUNCTION(BlueprintCallable, Category = "DebugMenu Plugin")
	void HideDebugMenu();

	// Checks if the debug menu is currently active
	UFUNCTION(BlueprintCallable, Category = "DebugMenu Plugin")
	bool IsDebugMenuActive() const;

	// Event triggered when the debug menu is shown
	UPROPERTY(BlueprintAssignable, Category = "DebugMenu Plugin")
	FDebugMenuActivationSignature OnDebugMenuShown;

	// Event triggered when the debug menu is hidden
	UPROPERTY(BlueprintAssignable, Category = "DebugMenu Plugin")
	FDebugMenuActivationSignature OnDebugMenuHidden;

protected:
	void InitializeMenuManager();
	int32 AddChildToHierarchy(FName ChildName, int32 ParentId);

	// libraries that will be automatically added to the menu
	// so that you don't have to place them in the map yourself
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Entries")
	TArray< TSubclassOf<ADebugMenuLibrary> > DefaultLibraries;

	// Background color of entries in the menu
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Display Options")
	FLinearColor MenuBGColor = FLinearColor(0.0f, 0.0f, 0.0f, 0.2f);

	// Background color of a highlighted entry in the menu
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Display Options")
	FLinearColor MenuHighlightedBGColor = FLinearColor(0.33f, 0.25f, 0.33f, 0.6f);

	// Margin for each row in the menu
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Display Options")
	FMargin MenuRowMargin = FMargin(1.0f, 1.0f, 1.0f, 1.0f);

	// Width of the menu
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Display Options")
	float MenuWidth = 500.0f;

	// Font for normal entries in the menu
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Display Options")
	FSlateFontInfo EntryFont;

	// Font for category entries in the menu
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Display Options")
	FSlateFontInfo CategoryFont;

	// Display menu on the right side of the screen, or left
	UPROPERTY(EditDefaultsOnly, Category = "DebugMenu Display Options")
	bool bDisplayOnRightSide = true;

	// Actors spawned from the "Default Libraries" property
	UPROPERTY(Transient)
	TArray<ADebugMenuLibrary*> SpawnedLibraries;

	// this is a tree structure, each node points to children and parent
	// first element is the root
	TArray< FDebugMenuHierarchyNode > DebugMenuHierarchy;

	int32 CurrentHierarchyIndex = 0;
	bool bHasInitialized = false;
	TSharedPtr<class SDebugMenuWidget> DebugMenuWidget;

	TArray<int32> SelectionCrumbs;

	TArray<FDebugMenuObjPropWatch> PropertyWatches;
	FSlateColorBrush WatchedBackgroundBrush = FSlateColorBrush(FLinearColor(1.0f, 1.0f, 0.0f, 0.2f));
};
