// Copyright (c) 2022 FWORKS. All Rights Reserved.

#pragma once

#include "CoreMinimal.h"
#include "Slate.h"

class ADebugMenuLibrary;
class UDebugMenuEntry;
class UInputComponent;
class ADebugMenuManager;
class FDebugMenuListEntryData;

class SDebugMenuWidget : public SCompoundWidget
{
public:
	SLATE_BEGIN_ARGS(SDebugMenuWidget)
	{}
	SLATE_ARGUMENT(bool, bRightSide)
	SLATE_ARGUMENT(TWeakObjectPtr<UWorld>, MenuWorld)
	SLATE_ARGUMENT(FLinearColor, MenuBGColor)
	SLATE_ARGUMENT(FLinearColor, MenuHighlightedBGColor)
	SLATE_ARGUMENT(FMargin, MenuRowMargin)
	SLATE_ARGUMENT(float, MenuWidth)
	SLATE_ARGUMENT(FSlateFontInfo, EntryFont)
	SLATE_ARGUMENT(FSlateFontInfo, CategoryFont)
	SLATE_END_ARGS()

public:
	void Construct(const FArguments& Args);
	~SDebugMenuWidget();
	void RefreshEntries(bool bSelectCrumb = false);
	void RequestHideMenu();

	void SelectUp();
	void SelectDown();
	TSharedPtr<FDebugMenuListEntryData> GetSelectedEntry() const;
	int32 GetSelectedIndex() const;

	TSharedPtr< SBox > ListContainer;

	TSharedPtr< SListView<TSharedPtr<FDebugMenuListEntryData>> > ListView;
	FTableViewStyle TransparentTableViewStyle;
	FTableRowStyle TransparentTableRowStyle;

	TWeakObjectPtr<UWorld> AssociatedWorld;
	TArray< TSharedPtr<FDebugMenuListEntryData> > DisplayedEntries;
	TWeakObjectPtr<ADebugMenuManager> AssociatedMenuManager;

	FLinearColor MenuBGColor = FLinearColor(0.0f, 0.0f, 0.0f, 0.2f);
	FLinearColor MenuHighlightedBGColor = FLinearColor(0.7f, 0.7f, 0.7f, 0.6f);
	FMargin MenuRowMargin = FMargin(1.0f, 1.0f, 1.0f, 1.0f);
	float MenuWidth = 500.0f;
	FSlateFontInfo EntryFont;
	FSlateFontInfo CategoryFont;

protected:
	/** Called to generate a widget for the specified tree item */
	TSharedRef<ITableRow> OnGenerateRow(TSharedPtr<FDebugMenuListEntryData> Item, const TSharedRef<STableViewBase>& Owner);

	void SetSelectedListItem(TSharedPtr<FDebugMenuListEntryData> Item);
};

class FDebugMenuListEntryData
{
public:
	FName CategoryDisplayName;
	TArray<TWeakObjectPtr<ADebugMenuLibrary>> LibrariesData;
	TWeakObjectPtr<UDebugMenuEntry> EntryData;

	TArray<TWeakPtr<SWidget>> FocusWidgetsOnConfirm;
	int32 FocusWidgetIndex = 0;

	bool bIsGoBack = false;

	void SetFocusWidgetOnConfirm(TWeakPtr<SWidget> NewWidget)
	{
		FocusWidgetsOnConfirm.Empty();
		FocusWidgetsOnConfirm.Add(NewWidget);
	}

	TWeakPtr<SWidget> GetCurrentFocusWidget()
	{
		if (FocusWidgetIndex < FocusWidgetsOnConfirm.Num())
		{
			return FocusWidgetsOnConfirm[FocusWidgetIndex];
		}

		TWeakPtr<SWidget> EmptyWidget;
		return EmptyWidget;
	}

	void IncrementFocusWidget()
	{
		if (FocusWidgetIndex >= FocusWidgetsOnConfirm.Num() - 1)
		{
			FocusWidgetIndex = 0;
		}
		else
		{
			FocusWidgetIndex++;
		}
	}

	void DecrementFocusWidget()
	{
		if (FocusWidgetIndex == 0)
		{
			if (FocusWidgetsOnConfirm.Num() > 0)
			{
				FocusWidgetIndex = FocusWidgetsOnConfirm.Num() - 1;
			}
		}
		else
		{
			FocusWidgetIndex--;
		}
	}
};

class SDebugMenuEntry : public SCompoundWidget
{
public:
	SLATE_BEGIN_ARGS(SDebugMenuEntry)
	{}
	SLATE_ARGUMENT(TSharedPtr<FDebugMenuListEntryData>, EntryData)
	SLATE_ARGUMENT(TWeakObjectPtr<UWorld>, MenuWorld)
	SLATE_ARGUMENT(float, MenuWidth)
	SLATE_ARGUMENT(FSlateFontInfo, EntryFont)
	SLATE_ARGUMENT(FSlateFontInfo, CategoryFont)
	SLATE_END_ARGS()

public:
	void Construct(const FArguments& args);
	~SDebugMenuEntry();

	virtual FReply OnMouseButtonDown(const FGeometry& MyGeometry, const FPointerEvent& MouseEvent) override;

private:
	TSharedPtr<FDebugMenuListEntryData> AssociatedEntryData;
	TWeakObjectPtr<UWorld> AssociatedWorld;
	FTextBlockStyle TransparentTextStyle;

public:
private:

};